<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\TmpMeterReading;
use App\Models\ReadingReport;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;

class TmpMeterReadingController extends Controller
{
    /**
     * Update tmp_meterreading records.
     */
    public function update(Request $request): JsonResponse
    {
        try {
            $postData = $request->json()->all();

            // Debug: Log the entire incoming JSON data
            \Log::info("Received JSON data: " . json_encode($postData));

            $response = [];

            foreach ($postData as $readingData) {
                // Debug: Log each reading data
                \Log::info("Processing reading data: " . json_encode($readingData));

                // Start transaction for atomic operations
                DB::beginTransaction();

                $tmpMeterUpdateSuccess = false;

                try {
                    // Validate required data and date format
                    $validator = Validator::make($readingData, [
                        'customerId' => 'required|integer',
                        'propId' => 'required|integer',
                        'companyId' => 'required|integer',
                        'currentReading' => 'required|integer',
                        'readingDate' => 'required|date',
                        'comments' => 'nullable|string',
                        'imagePath' => 'nullable|string',
                    ]);

                    if ($validator->fails()) {
                        $response[] = [
                            'message' => 'Validation failed: ' . implode(', ', $validator->errors()->all()),
                            'status' => 'error'
                        ];
                        DB::rollback();
                        continue;
                    }

                    // Update tmp_meterreading table
                    $updatedRows = TmpMeterReading::where('CustomerID', $readingData['customerId'])
                        ->where('PropID', $readingData['propId'])
                        ->where('CompanyID', $readingData['companyId'])
                        ->update([
                            'Reading' => $readingData['currentReading'],
                            'ReadingDate' => $readingData['readingDate'],
                            'ReadingComments' => $readingData['comments'] ?? null,
                            'ImagePath' => $readingData['imagePath'] ?? null,
                        ]);

                    if ($updatedRows > 0) {
                        $tmpMeterUpdateSuccess = true;
                        \Log::info(
                            "tmp_meterreading updated successfully for CustomerID: " .
                            $readingData['customerId'] . ", PropID: " . $readingData['propId'] .
                            ", CompanyID: " . $readingData['companyId'] .
                            ". Affected rows: " . $updatedRows
                        );
                    } else {
                        \Log::info("No record found in tmp_meterreading with CustomerID: " . $readingData['customerId'] .
                            ", PropID: " . $readingData['propId'] . ", CompanyID: " . $readingData['companyId']);
                    }

                    // Commit transaction
                    DB::commit();

                    // Prepare response
                    if ($tmpMeterUpdateSuccess) {
                        $response[] = [
                            'message' => 'tmp_meterreading updated successfully',
                            'status' => 'success'
                        ];
                    } else {
                        $response[] = [
                            'message' => 'No records updated or missing required data',
                            'status' => 'info'
                        ];
                    }

                } catch (\Exception $e) {
                    // Rollback transaction on error
                    DB::rollback();

                    $response[] = [
                        'message' => $e->getMessage(),
                        'status' => 'error'
                    ];
                    \Log::error("Error: " . $e->getMessage());
                }
            }

            return response()->json($response);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An error occurred: ' . $e->getMessage(),
                'status' => 'error'
            ], 500);
        }
    }

    /**
     * Upload image for meter reading.
     */
    public function uploadImage(Request $request): JsonResponse
    {
        try {
            // Validate the uploaded file
            $validator = Validator::make($request->all(), [
                'file' => 'required|file|image|mimes:jpeg,png,jpg,gif|max:2048', // 2MB max
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed: ' . implode(', ', $validator->errors()->all())
                ], 400);
            }

            // Check if file is uploaded
            if (!$request->hasFile('file')) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No file uploaded'
                ], 400);
            }

            $file = $request->file('file');

            // Use the original filename as provided by Android
            $filename = $file->getClientOriginalName();

            // Define the upload directory
            $uploadDir = public_path('meterreadingimages');

            // Ensure the directory exists
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            // Move the file to the public directory
            $filePath = $uploadDir . DIRECTORY_SEPARATOR . $filename;
            if (!$file->move($uploadDir, $filename)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to move the uploaded file'
                ], 500);
            }

            // Get the web accessible path
            $webPath = '/meterreadingimages/' . $filename;

            \Log::info("Image uploaded successfully: " . $webPath);

            return response()->json([
                'status' => 'success',
                'message' => 'Image uploaded successfully',
                'file_path' => $webPath,
                'file_name' => $filename,
                'full_path' => $filePath
            ]);

        } catch (\Exception $e) {
            \Log::error("Image upload error: " . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a reading report.
     */
    public function storeReadingReport(Request $request): JsonResponse
    {
        try {
            $data = $request->json()->all();

            // Validate required data
            $validator = Validator::make($data, [
                'company_id' => 'required|integer',
                'prop_id' => 'required|integer',
                'readingCodeDfn' => 'nullable|string',
                'meterNumber' => 'nullable|string',
                'accountNumber' => 'nullable|string',
                'meterReading' => 'nullable|string',
                'comments' => 'nullable|string',
                'imagePath' => 'required|string',
                'reportDate' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'Validation failed: ' . implode(', ', $validator->errors()->all()),
                    'status' => 'error'
                ], 400);
            }

            // Insert into reading_reports table
            $readingReport = ReadingReport::create([
                'company_id' => $data['company_id'],
                'prop_id' => $data['prop_id'],
                'readingCodeDfn' => $data['readingCodeDfn'] ?? null,
                'meterNumber' => $data['meterNumber'] ?? null,
                'accountNumber' => $data['accountNumber'] ?? null,
                'meterReading' => $data['meterReading'] ?? null,
                'comments' => $data['comments'] ?? null,
                'imagePath' => $data['imagePath'],
                'reportDate' => $data['reportDate'],
            ]);

            return response()->json([
                'message' => 'Reading report stored successfully',
                'status' => 'success',
                'data' => $readingReport
            ]);

        } catch (\Exception $e) {
            \Log::error("Store reading report error: " . $e->getMessage());
            return response()->json([
                'message' => 'An error occurred: ' . $e->getMessage(),
                'status' => 'error'
            ], 500);
        }
    }
}
