<?php

namespace App\Http\Controllers;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Company;

class BillDisplaySummaryController extends Controller
{
 
public function viewPrintBillsPerCourt($billingPeriod)
{
    // Get the authenticated user's CompanyID
    $companyId = $this->user()->CompanyID;

    if (!$companyId) {
        return redirect()->back()->with('error', 'No company associated with this account.');
    }

    // Fetch company details
    $company = Company::find($companyId);
    if (!$company) {
        return redirect()->back()->with('error', 'Company not found.');
    }

    // Use VIEW_BILL_COURTS from the migration file
    $bills = DB::table('view_bill_courts')
        ->where('BillingPeriod', $billingPeriod)
        ->orderBy('Court', 'asc')
        ->get();

    // Group bills by court for the view
    $groupedBills = $bills->groupBy('TenantName');

    // Calculate totals for the summary
    $totalConsumption = $bills->sum('ConsumptionBilled');
    $totalBill = $bills->sum('TotalBill');

    // Get bill line details for the billing period
    $billLines = DB::table('billline as bl')
        ->join('Bill as b', 'bl.BillID', '=', 'b.BillID')
        ->where('b.BillingPeriod', $billingPeriod)
        ->select(
            'bl.BillID',
            'bl.BLineID',
            'bl.BLineDesc',
            'bl.BLineUnits',
            'bl.BLineUnitPrice',
            'bl.BLineAmount',
            'bl.BLineCategID',
            'bl.BLineTLineID',
            'b.BillingPeriod'
        )
        ->orderBy('bl.BillID')
        ->orderBy('bl.BLineID')
        ->get();

    // Group bill lines by BillID for easier access in the view
    $groupedBillLines = $billLines->groupBy('BillID');

    // Add AllBLineDescriptions and AllBLineAmounts to each bill
    // Since the view aggregates by customer, we need to find all bills for each customer
    $bills = $bills->map(function ($bill) use ($billingPeriod) {
        // Find the customer by ClientCode
        $customer = DB::table('customer')->where('ClientCode', $bill->ClientCode)->first();

        if ($customer) {
            // Get all bills for this customer in the billing period
            $customerBills = DB::table('bill')
                ->where('CustomerID', $customer->CustomerID)
                ->where('BillingPeriod', $billingPeriod)
                ->pluck('BillID');

            if ($customerBills->count() > 0) {
                // Get bill lines for all bills of this customer
                $billLinesForCustomer = DB::table('billline')
                    ->whereIn('BillID', $customerBills)
                    ->orderBy('BillID')
                    ->orderBy('BLineID')
                    ->get();

                $descriptions = $billLinesForCustomer->pluck('BLineDesc')->toArray();
                $amounts = $billLinesForCustomer->pluck('BLineAmount')->toArray();

                $bill->AllBLineDescriptions = implode(', ', $descriptions);
                $bill->AllBLineAmounts = implode(', ', $amounts);
            } else {
                $bill->AllBLineDescriptions = '';
                $bill->AllBLineAmounts = '';
            }
        } else {
            $bill->AllBLineDescriptions = '';
            $bill->AllBLineAmounts = '';
        }

        return $bill;
    });

    // Get all unique billing periods for the filter
    $billingPeriods = DB::table('view_bill_courts')
        ->select('BillingPeriod')
        ->distinct()
        ->orderBy('BillingPeriod', 'desc')
        ->pluck('BillingPeriod');

    return view('reports.print-bills-per-court', [
        'bills' => $bills,
        'groupedBills' => $groupedBills,
        'billingPeriod' => $billingPeriod,
        'billingPeriods' => $billingPeriods,
        'totalConsumption' => $totalConsumption,
        'totalBill' => $totalBill,
        'billLines' => $billLines,
        'groupedBillLines' => $groupedBillLines,
        'company' => $company
    ]);
}


 


}
