<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\ProUnit;
use App\Models\PropService;
use App\Models\UnitAssignment;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\CustomerStatement;
use Barryvdh\DomPDF\Facade\Pdf;

class CustomerController extends Controller
{
    /**
     * Display a listing of the customers.
     */
    public function index(Request $request)
    {
        // Get the authenticated user's CompanyID
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        // Get the status filter from the request
        $status = $request->query('status');
        $search = $request->query('search');
        $propertyId = $request->query('property_id');
        
        // Start building the query
        $query = Customer::where('CompanyID', $companyId);
        
        // Apply search filter if provided
        if (!empty($search)) {
            $query->where(function($q) use ($search) {
                $q->where('CustomerName', 'LIKE', '%' . $search . '%')
                  ->orWhere('AccountNo', 'LIKE', '%' . $search . '%')
                  ->orWhere('Email', 'LIKE', '%' . $search . '%')
                  ->orWhere('CustomerEmail', 'LIKE', '%' . $search . '%')
                  ->orWhere('TelNo', 'LIKE', '%' . $search . '%');
            });
        }
        
        // Apply status filter if provided
        if (!empty($status) && in_array($status, ['A', 'I'])) {
            $query->where('AccountStatus', $status);
        }

        // Apply property filter if provided
        if (!empty($propertyId)) {
            $query->whereHas('unitAssignments', function($q) use ($propertyId) {
                $q->whereHas('unit', function($unitQuery) use ($propertyId) {
                    $unitQuery->where('PropID', $propertyId);
                });
            });
        }
        
        $customers = $query->orderBy('CustomerID')
            ->paginate(15)
            ->withQueryString();
        
        // Get the count of customers by status for the filter
        $statusCounts = [
            'all' => Customer::where('CompanyID', $companyId)->count(),
            'A' => Customer::where('CompanyID', $companyId)->where('AccountStatus', 'A')->count(),
            'I' => Customer::where('CompanyID', $companyId)->where('AccountStatus', 'I')->count(),
        ];

        // Get unique properties for the filter dropdown (filtered by company)
        $properties = \App\Models\Property::where('CompanyID', $companyId)
            ->orderBy('PropName')
            ->pluck('PropName', 'PropID');

        return view('customers.index', compact('customers', 'status', 'statusCounts', 'search', 'propertyId', 'properties'));
    }

    /**
     * Show the form for creating a new customer.
     */
    public function create()
    {
        // Get the authenticated user's CompanyID
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        // Get all available units for the unit dropdown, filtered by company
        $units = ProUnit::with('property')
            ->where('UnitActive', 1) // Only active units
            ->whereHas('property', function($query) use ($companyId) {
                $query->where('CompanyID', $companyId);
            })
            ->where(function($query) {
                $query->whereDoesntHave('currentAssignments')
                    ->orWhereIn('UnitID', function($q) {
                        $q->select('UnitID')
                            ->from('customer')
                            ->whereNull('AccountStatus')
                            ->orWhere('AccountStatus', '!=', 'A');
                    });
            })
            ->get()
            ->mapWithKeys(function($unit) {
                return [
                    $unit->UnitID => [
                        'identity' => $unit->UnitIdentity . ' - ' . $unit->property->PropName,
                        'property_id' => $unit->PropID,
                        'water_meter' => $unit->WMeterNo,
                        'electricity_meter' => $unit->EMeterNo
                    ]
                ];
            });

        // Get all active services for the company
        $services = \App\Models\PropService::where('CompanyID', $companyId)
            // ->where('IsActive', 1)
            ->orderBy('ServiceName')
            ->get();

        return view('customers.create', compact('units', 'services'));
    }
    
    /**
     * Get meter details for a specific unit
     */
    public function getUnitDetails(ProUnit $unit)
    {
        try {
            // Get the authenticated user's CompanyID
            $companyId =$this->user()->CompanyID;
            
            if (!$companyId) {
                return response()->json([
                    'success' => false,
                    'message' => 'No company associated with this account.'
                ], 403);
            }
            
            // Verify the unit belongs to the user's company
            if ($unit->property->CompanyID != $companyId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to unit details.'
                ], 403);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'WMeterNo' => $unit->WMeterNo,
                    'WLastReading' => $unit->WLastReading,
                    'EMeterNo' => $unit->EMeterNo,
                    'ELastReading' => $unit->ELastReading,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch unit details.'
            ], 500);
        }
    }

     /**
     * Get services for a specific Property unit
     */
    public function getServicesByProperty($propertyId)
    {
        try {
           
            // Get the authenticated user's CompanyID
            $companyId = $this->user()->CompanyID;
            
            if (!$companyId) {
                return response()->json([
                    'success' => false,
                    'message' => 'No company associated with this account.'
                ], 403);
            }
            
            // Verify the property belongs to the user's company
            $property = \App\Models\Property::where('PropID', $propertyId)
                ->where('CompanyID', $companyId)
                ->first();
                
            if (!$property) {
                return response()->json([
                    'success' => false,
                    'message' => 'Property not found or you do not have permission to access it.'
                ], 404);
            }
            
            // Get active services for the property
            $services = \App\Models\PropService::where('PropID', $propertyId)
            ->where('CompanyID',$companyId)
               
                ->orderBy('ServiceName')
                ->get(['IDService', 'ServiceName','Price']);
                //  dd($services);

            return response()->json([
                'success' => true,
                'data' => $services
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error fetching property services: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch services. Please try again.'
            ], 500);
        }
    }

    /**
     * Store a newly created customer in storage.
     */
    public function store(Request $request)
    {
         $companyId =$this->user()->CompanyID;
        $validated = $request->validate([
            'CustomerName' => 'required|string|max:100',
            'AccountNo' => 'required|string|max:50|unique:customer,AccountNo',
            'TelNo' => 'required|string|max:20',
            'Email' => 'nullable|email|max:100',
            'Address1' => 'required|string|max:100',
            'Address2' => 'nullable|string|max:100',
            'Address3' => 'nullable|string|max:50',
            'ClientCode' => 'nullable|string|max:50',
            'TenantName' => 'nullable|string|max:100',
            'AccountType' => 'required|in:RESIDENTIAL,COMMERCIAL',
            'ContractDate' => 'required|date',
            'AccountStatus' => 'required|in:A,I',
            'Deposit' => 'nullable|numeric|min:0',
            'DepositDate' => 'nullable|date|required_with:Deposit',
            'WMeterNo' => 'nullable|string|max:50',
            'WInitialReading' => 'nullable|numeric|min:0',
            'EMeterNo' => 'nullable|string|max:50',
            'EInitialReading' => 'nullable|numeric|min:0',
            'UnitID' => 'nullable|exists:prounits,UnitID',
            'CompanyID' => 'nullable|exists:Company,CompanyID',
        ]);
        
        $validated['CompanyID']=$companyId;

         // Set default AccountStatus to 'I' (Inactive)
        $validated['AccountStatus'] = 'I';
        
        // If UnitID is provided, set status to 'A' (Active)
        if (!empty($validated['UnitID'])) {
            $validated['AccountStatus'] = 'A';
        }
        
        // Address Line and Account File Name
        $addressLine = implode(', ', array_filter([$validated['TelNo'], $validated['Address1'], $validated['Address2'], $validated['Address3']]));
        $validated['AccountFileName'] = implode(' - ', array_filter([$validated['Address1'], $validated['TelNo']]));
        // Get the authenticated user's CompanyID
       
         $validated['addressLine']=$addressLine;

        Log::info("Details to be saved: " . json_encode($validated, JSON_PRETTY_PRINT));

        // Format phone number (remove any non-numeric characters)
        if (isset($validated['TelNo'])) {
            $validated['TelNo'] = preg_replace('/\D/', '', $validated['TelNo']);
        }

        DB::beginTransaction();
        try {
            $customer = Customer::create($validated);
            
            // If a unit is assigned, update the unit's occupation status and create assignment
            if (!empty($validated['UnitID'])) {
                $unit = ProUnit::find($validated['UnitID']);
                if ($unit) {
                    $unit->OccupationStatus = 'Active';
                    $unit->save();
                    
                    // Create unit assignment record
                    UnitAssignment::create([
                        'customer_id' => $customer->CustomerID,
                        'unit_id' => $validated['UnitID'],
                        'start_date' => now(),
                        'status' => 'active'
                    ]);
                }
            }
            
            // Handle services assignment
            if ($request->has('services')) {
                foreach ($request->input('services') as $serviceId => $serviceData) {
                    if (isset($serviceData['id']) && $serviceData['id']) { // If service is checked
                        // Get quantity (default to 1 if not provided or invalid)
                        $quantity = max(1, intval($serviceData['quantity'] ?? 1));
                        
                        // Convert price to float and ensure it's a valid number
                        $unitPrice = is_numeric($serviceData['price'] ?? 0) ? (float)$serviceData['price'] : 0;
                        
                        // Calculate total price (unit price * quantity)
                        $totalPrice = $unitPrice * $quantity;
                        
                        // Ensure start date is valid, default to today if not
                        $startDate = !empty($serviceData['start_date']) ? $serviceData['start_date'] : now()->format('Y-m-d');
                        
                        // Get the service to ensure it exists
                        $service = \App\Models\PropService::find($serviceId);
                        
                        if ($service) {
                            $customer->propServices()->attach($serviceId, [
                                'Price' => $unitPrice,
                                'Quantity' => $quantity,
                                'StartDate' => $startDate,
                                'EndDate' => null, // No end date initially
                                'IsActive' => 1,
                                'Notes' => 'Added during customer creation',
                                'created_at' => now(),
                                'updated_at' => now()
                            ]);
                            
                            // Log the attachment for debugging
                            \Log::info("Attached service to customer", [
                                'customer_id' => $customer->CustomerID,
                                'service_id' => $serviceId,
                                'unit_price' => $unitPrice,
                                'quantity' => $quantity,
                                'calculated_total' => $totalPrice,
                                'start_date' => $startDate
                            ]);
                        } else {
                            \Log::warning("Attempted to attach non-existent service to customer", [
                                'customer_id' => $customer->CustomerID,
                                'service_id' => $serviceId
                            ]);
                        }
                    }
                }
            }
            
            DB::commit();
            return redirect()->route('customers.show', $customer->CustomerID)
                ->with('success', 'Customer created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating customer: ' . $e->getMessage());
            return back()->withInput()->with('error', 'Error creating customer. Please try again. ' . $e->getMessage());
        }
    }

    /**
     * Display the specified customer.
     */
    public function show(Customer $customer, Request $request)
    {
        // Clear any flash messages if this is a direct visit (not a redirect)
        if (!$request->session()->has('_previous')) {
            $request->session()->forget(['success', 'error']);
        }

        $customer->load([
            'unit.property',
            'bills' => function($query) {
                $query->orderBy('BillingDate', 'desc')->take(10);
            },
            'payments' => function($query) {
                $query->orderBy('PaymentDate', 'desc')->take(10);
            },
            'unitAssignments' => function($query) {
                $query->with('unit.property')
                    ->orderBy('start_date', 'desc');
            },
            'propServices' => function($query) {
                $query->withPivot(['Price', 'StartDate', 'EndDate', 'IsActive', 'Notes']);
            }
        ]);

        // Generate a 3-month statement for the dashboard
        $endDate = now()->format('Y-m-d');
        $startDate = now()->subMonths(3)->format('Y-m-d');
        $statement = $customer->generateAccountStatement($startDate, $endDate);

        // Load available services for assignment
        $availableServices = PropService::whereNotIn('IDService', $customer->propServices->pluck('IDService'))
            ->get();

        return view('customers.show', compact('customer', 'availableServices', 'statement'));
    }

    /**
     * Show the form for editing the specified customer.
     */
    public function edit(Customer $customer)
    {
        // Get all available units for the unit dropdown, including the customer's current unit
        $units = ProUnit::with('property')
            ->whereDoesntHave('currentAssignments')
            ->orWhere('UnitID', $customer->UnitID) // Include the customer's current unit
            ->orWhereIn('UnitID', function($query) {
                $query->select('UnitID')
                    ->from('customer')
                    ->whereNull('AccountStatus')
                    ->orWhere('AccountStatus', '!=', 'A');
            })
            ->get()
            ->mapWithKeys(function($unit) {
                return [
                    $unit->UnitID => [
                        'text' => $unit->UnitIdentity . ' - ' . $unit->property->PropName,
                        'data-property-id' => $unit->PropID,
                        'data-water-meter' => $unit->WMeterNo,
                        'data-electricity-meter' => $unit->EMeterNo
                    ]
                ];
            });
            
        // Load customer's current services with pivot data
        $customer->load(['propServices' => function($query) {
            $query->withPivot(['Price', 'Quantity', 'StartDate', 'IsActive']);
        }]);
            
        return view('customers.edit', compact('customer', 'units'));
    }

    /**
     * Update the specified customer in storage.
     */
    public function update(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'CustomerName' => 'required|string|max:100',
            'AccountNo' => 'required|string|max:50|unique:customer,AccountNo,' . $customer->CustomerID . ',CustomerID',
            'TelNo' => 'required|string|max:20',
            'Email' => 'nullable|email|max:100',
            'Address1' => 'required|string|max:100',
            'Address2' => 'nullable|string|max:100',
            'Address3' => 'nullable|string|max:50',
            'ClientCode' => 'nullable|string|max:50',
            'TenantName' => 'nullable|string|max:100',
            'AccountType' => 'required|in:RESIDENTIAL,COMMERCIAL',
            'ContractDate' => 'required|date',
            'AccountStatus' => 'required|in:A,I',
            'Deposit' => 'nullable|numeric|min:0',
            'DepositDate' => 'nullable|date|required_with:Deposit',
            'WMeterNo' => 'nullable|string|max:50',
            'WInitialReading' => 'nullable|numeric|min:0',
            'EMeterNo' => 'nullable|string|max:50',
            'EInitialReading' => 'nullable|numeric|min:0',
            'UnitID' => 'nullable|exists:prounits,UnitID',
        ]);

        
        // Format phone number (remove any non-numeric characters)
        if (isset($validated['TelNo'])) {
            $validated['TelNo'] = preg_replace('/\D/', '', $validated['TelNo']);
        }

        DB::beginTransaction();
        try {
            $previousUnitId = $customer->UnitID;
            $newUnitId = $validated['UnitID'] ?? null;
            
            // Apply all validated data to the customer model first
            $customer->fill($validated);
            
            // Force update AccountStatus even if it hasn't changed
            if (isset($validated['AccountStatus'])) {
                // Use setAttribute to ensure the attribute is marked as dirty
                $customer->setAttribute('AccountStatus', $validated['AccountStatus']);
            }
            
            // Debug: Log the customer data after fill
            \Log::info('Customer data after fill:', [
                'AccountStatus' => $customer->AccountStatus,
                'original_AccountStatus' => $customer->getOriginal('AccountStatus'),
                'isDirty_AccountStatus' => $customer->isDirty('AccountStatus'),
                'all_dirty' => $customer->getDirty()
            ]);
            
            // Handle unit assignment changes first
            if ($previousUnitId != $newUnitId) {
                // Update the customer's UnitID
                $customer->UnitID = $newUnitId;
                // If unit was changed, update the previous unit's status and end active assignments
                if ($previousUnitId) {
                    $previousUnit = ProUnit::find($previousUnitId);
                    if ($previousUnit) {
                        $previousUnit->OccupationStatus = 'InActive';
                        $previousUnit->save();
                        
                        // First, check if there's already a terminated assignment for this customer
                        $existingTerminated = UnitAssignment::where('customer_id', $customer->CustomerID)
                            ->where('status', 'terminated')
                            ->first();
                            
                        if ($existingTerminated) {
                            // Update the existing terminated assignment
                            $existingTerminated->update([
                                'unit_id' => $previousUnitId,
                                'end_date' => now(),
                                'notes' => 'Unit changed to ' . ($newUnitId ?: 'none')
                            ]);
                            
                            // End the current active assignment
                            UnitAssignment::where('unit_id', $previousUnitId)
                                ->where('customer_id', $customer->CustomerID)
                                ->where('status', 'active')
                                ->delete();
                        } else {
                            // End any active assignments for the previous unit
                            UnitAssignment::where('unit_id', $previousUnitId)
                                ->where('customer_id', $customer->CustomerID)
                                ->where('status', 'active')
                                ->update([
                                    'status' => 'terminated',
                                    'end_date' => now(),
                                    'notes' => 'Unit changed to ' . ($newUnitId ?: 'none')
                                ]);
                        }
                    }
                }

                
                
                // Update the new unit's status and create new assignment
                if ($newUnitId) {
                    $newUnit = ProUnit::find($newUnitId);
                    if ($newUnit) {
                        $newUnit->OccupationStatus = 'Active';
                        $newUnit->save();
                        
                        // End any existing active assignments for this customer
                        UnitAssignment::where('customer_id', $customer->CustomerID)
                            ->where('status', 'active')
                            ->update([
                                'status' => 'transferred',
                                'end_date' => now(),
                                'notes' => 'Transferred to another unit'
                            ]);
                        
                        // Create new assignment
                        UnitAssignment::create([
                            'customer_id' => $customer->CustomerID,
                            'unit_id' => $newUnitId,
                            'start_date' => now(),
                            'status' => 'active',
                            'notes' => 'Assigned during customer update'
                        ]);
                    }
                } else {
                    // If no new unit is selected, set UnitID to null
                    $customer->UnitID = null;
                    $customer->save();
                }
            }
            
            // Save all customer data including AccountStatus
            $saved = $customer->save();
            // dd( $customer);
            // Debug: Log the saved status and current AccountStatus
            \Log::info('Customer save status:', [
                'saved' => $saved,
                'AccountStatus' => $customer->AccountStatus,
                'wasRecentlyCreated' => $customer->wasRecentlyCreated,
                'changes' => $customer->getChanges()
            ]);
            
            // Handle service assignments
            if ($request->has('services') && is_array($request->services)) {
                $servicesToSync = [];
                
                foreach ($request->services as $serviceId => $serviceData) {
                    if (isset($serviceData['id'])) {
                        $servicesToSync[$serviceId] = [
                            'Price' => $serviceData['price'] ?? 0,
                            'Quantity' => $serviceData['quantity'] ?? 1,
                            'StartDate' => $serviceData['start_date'] ?? now()->format('Y-m-d')
                        ];
                    }
                }
                
                // Sync the services with pivot data
                $customer->propServices()->sync($servicesToSync);
                
                // Log the service update
                Log::info('Updated services for customer', [
                    'customer_id' => $customer->CustomerID,
                    'services' => array_keys($servicesToSync)
                ]);
            } else {
                // If no services are selected, detach all services
                $customer->propServices()->detach();
                Log::info('Removed all services for customer', ['customer_id' => $customer->CustomerID]);
            }
            
            DB::commit();
            return redirect()->route('customers.show', $customer->CustomerID)
                ->with('success', 'Customer updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating customer: ' . $e->getMessage());
            return back()->withInput()->with('error', 'Error updating customer. Please try again. ' . $e->getMessage());
        }
    }

    /**
     * Display the customer's account statement.
     */
    public function statement(Request $request, Customer $customer)
    {
        $dateRanges = [
            'today' => 'Today',
            'yesterday' => 'Yesterday',
            'this_week' => 'This Week',
            'last_week' => 'Last Week',
            'this_month' => 'This Month',
            'last_month' => 'Last Month',
            'last_3_months' => 'Last 3 Months',
            'last_6_months' => 'Last 6 Months',
            'this_year' => 'This Year',
            'last_year' => 'Last Year',
            'custom' => 'Custom Range'
        ];
        
        $selectedRange = $request->input('range', 'last_3_months');
        
        // Set date range based on selection
        $endDate = now();
        
        switch ($selectedRange) {
            case 'today':
                $startDate = now()->startOfDay();
                break;
            case 'yesterday':
                $startDate = now()->subDay()->startOfDay();
                $endDate = now()->subDay()->endOfDay();
                break;
            case 'this_week':
                $startDate = now()->startOfWeek();
                break;
            case 'last_week':
                $startDate = now()->subWeek()->startOfWeek();
                $endDate = now()->subWeek()->endOfWeek();
                break;
            case 'this_month':
                $startDate = now()->startOfMonth();
                break;
            case 'last_month':
                $startDate = now()->subMonth()->startOfMonth();
                $endDate = now()->subMonth()->endOfMonth();
                break;
            case 'last_3_months':
                $startDate = now()->subMonths(3)->startOfDay();
                break;
            case 'last_6_months':
                $startDate = now()->subMonths(6)->startOfDay();
                break;
            case 'this_year':
                $startDate = now()->startOfYear();
                break;
            case 'last_year':
                $startDate = now()->subYear()->startOfYear();
                $endDate = now()->subYear()->endOfYear();
                break;
            case 'custom':
                $startDate = $request->filled('start_date') 
                    ? \Carbon\Carbon::parse($request->input('start_date'))
                    : now()->subMonths(3);
                $endDate = $request->filled('end_date')
                    ? \Carbon\Carbon::parse($request->input('end_date'))
                    : now();
                break;
            default:
                $startDate = now()->subMonths(3);
        }
        
        // Ensure end date is not in the future
        $endDate = $endDate->isFuture() ? now() : $endDate;
        
        // Generate the statement
        $statement = $customer->generateAccountStatement(
            $startDate->format('Y-m-d'),
            $endDate->format('Y-m-d')
        );
        
        // Get company details
        $company = $this->user()->company;
        
        // Format company address
        $addressParts = array_filter([
            $company->Address1,
            $company->Address2,
            $company->City ?? null,
            $company->Country ?? null
        ]);
        
        // Get logo path - check both storage and public/logo directories
        $logoPath = null;
        if ($company->CompanyLogo) {
            // First check if file exists in public/logo
            $publicLogoPath = 'logo/' . $company->CompanyLogo;
            if (file_exists(public_path($publicLogoPath))) {
                $logoPath = asset($publicLogoPath);
            } 
            // Fallback to storage if not found in public/logo
            elseif (file_exists(storage_path('app/public/' . $company->CompanyLogo))) {
                $logoPath = asset('storage/' . $company->CompanyLogo);
            }
        }
        
        $companyData = [
            'name' => $company->CompanyName,
            'address' => implode(', ', $addressParts),
            'phone' => $company->TelNo,
            'email' => $company->Email,
            'logo' => $logoPath
        ];
        
        return view('customers.statement', [
            'customer' => $customer,
            'statement' => $statement,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'dateRanges' => $dateRanges,
            'selectedRange' => $selectedRange,
            'company' => $companyData
        ]);
    }
    
    /**
     * Export the customer's account statement as PDF.
     */
    public function exportStatementPdf(Customer $customer, Request $request)
    {
        $startDate = $request->input('start_date', now()->subMonths(3)->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        
        // Generate the statement
        $statement = $customer->generateAccountStatement($startDate, $endDate);
        
        // Format dates for the PDF
        $startDate = Carbon::parse($startDate);
        $endDate = Carbon::parse($endDate);
        
        // Get company information
        $company =  $this->user()->company;
        $addressParts = array_filter([$company->Address1, $company->Address2, $company->City ?? null, $company->Country ?? null]);
        
        // Find the logo file path
        $logoPath = null;
        if ($company->CompanyLogo) {
            // Check public/logo directory first
            $publicLogoPath = 'logo/' . $company->CompanyLogo;
            if (file_exists(public_path($publicLogoPath))) {
                $logoPath = public_path($publicLogoPath);
            } 
            // Check storage if not found in public/logo
            elseif (file_exists(storage_path('app/public/' . $company->CompanyLogo))) {
                $logoPath = storage_path('app/public/' . $company->CompanyLogo);
            }
        }
        
        // If no logo found, use default logo
        if (!$logoPath) {
            $logoPath = public_path('logo/sulispmslogo.png');
        }
        
        $companyData = [
            'name' => $company->CompanyName,
            'address' => implode(', ', $addressParts),
            'phone' => $company->TelNo,
            'email' => $company->Email,
            'logo' => $logoPath  // Use file system path for PDF generation
        ];
        
        $pdf = PDF::loadView('pdf.statement', [
            'customer' => $customer,
            'statement' => $statement,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'company' => $companyData
        ]);
        
        $filename = 'statement_' . $customer->AccountNo . '_' . $startDate->format('Y-m-d') . '_to_' . $endDate->format('Y-m-d') . '.pdf';
        
        return $pdf->download($filename);
    }
    
    /**
     * Remove the specified customer from storage.
     */
    /**
     * Email the customer's account statement.
     */
    public function emailStatement(Customer $customer, Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $startDate = Carbon::parse($request->start_date);
        $endDate = Carbon::parse($request->end_date);
        
        // Generate the statement
        $statement = $customer->generateAccountStatement(
            $startDate->format('Y-m-d'),
            $endDate->format('Y-m-d')
        );
        
        // Get company information
        $company =  $this->user()->company;
        $addressParts = array_filter([$company->Address1, $company->Address2, $company->City ?? null, $company->Country ?? null]);
        
        // Prepare company data
        $companyData = [
            'name' => $company->CompanyName,
            'address' => implode(', ', $addressParts),
            'phone' => $company->TelNo,
            'email' => $company->Email,
        ];

        try {
            // Send email
            Mail::to($customer->Email)
                ->send(new CustomerStatement($customer, $statement, $startDate, $endDate, $companyData));
                
            return back()->with('success', 'Statement has been sent to ' . $customer->Email);
        } catch (\Exception $e) {
            Log::error('Error sending statement email: ' . $e->getMessage());
            return back()->with('error', 'Failed to send email. Please try again.');
        }
    }

    /**
     * Remove the specified customer from storage.
     */
    public function destroy(Customer $customer)
    {
        // Prevent deletion if customer has associated records
        if ($customer->bills()->exists() || $customer->payments()->exists() || $customer->meterReadings()->exists()) {
            return back()->with('error', 'Cannot delete customer with associated bills, payments, or meter readings.');
        }

        DB::beginTransaction();
        try {
            // Update the unit's occupation status if assigned
            if ($customer->UnitID) {
                $unit = ProUnit::find($customer->UnitID);
                if ($unit) {
                    $unit->OccupationStatus = 'VACANT';
                    $unit->save();
                }
            }

            $customer->delete();
            DB::commit();
            
            return redirect()->route('customers.index')
                ->with('success', 'Customer deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting customer: ' . $e->getMessage());
            return back()->with('error', 'Error deleting customer. Please try again.');
        }
    }

    /**
     * Deactivate the specified customer.
     */
    public function deactivate(Customer $customer)
    {
        try {
            $customer->update(['AccountStatus' => 'I','UnitID'=>null]);
            
            return back()
                ->with('success', 'Customer deactivated successfully.');
                
        } catch (\Exception $e) {
            Log::error('Error deactivating customer: ' . $e->getMessage());
            
            return back()
                ->with('error', 'Failed to deactivate customer. Please try again.');
        }
    }

    /**
     * Activate the specified customer.
     */
    public function activate(Customer $customer)
    {
        try {
            // Check if customer has a UnitID assigned
            if (!$customer->UnitID) {
                return back()
                    ->with('sweet_alert', [
                        'icon' => 'error',
                        'title' => 'Cannot activate customer',
                        'text' => 'A unit must be assigned before activation.',
                        'confirmButtonText' => 'Dismiss'
                    ]);
            }
            
            $customer->update(['AccountStatus' => 'A']);
            
            return back()
                ->with('success', 'Customer activated successfully.');
                
        } catch (\Exception $e) {
            Log::error('Error activating customer: ' . $e->getMessage());
            
            return back()
                ->with('error', 'Failed to activate customer. Please try again.');
        }
    }

    /**
     * Show the form for managing customer's services.
     */
    public function services(Customer $customer)
    {
        $customer->load('propServices');
        $allServices = PropService::orderBy('ServiceName')->get();
        
        return view('customers.services', compact('customer', 'allServices'));
    }

    /**
     * Add a service to the customer.
     */
    public function addService(Request $request, Customer $customer)
    {
        // Check if customer has UnitID and is active
        if (!$customer->UnitID || $customer->AccountStatus !== 'A') {
            return back()
                ->with('sweet_alert', [
                    'icon' => 'error',
                    'title' => 'Cannot assign service',
                    'text' => 'Services can only be assigned to active customers with a unit assigned.',
                    'confirmButtonText' => 'Dismiss'
                ]);
        }

        $validated = $request->validate([
            'service_id' => 'required|exists:PropService,IDService',
            'price' => 'required|numeric|min:0',
            'quantity' => 'required|integer|min:1',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'notes' => 'nullable|string|max:500',
        ]);

        try {
            $customer->propServices()->attach($validated['service_id'], [
                'Price' => $validated['price'],
                'Quantity' => $validated['quantity'],
                'StartDate' => $validated['start_date'],
                'EndDate' => $validated['end_date'] ?? null,
                'Notes' => $validated['notes'] ?? null,
                'IsActive' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return redirect()->route('customers.services', $customer->CustomerID)
                ->with('success', 'Service added successfully.');
                
        } catch (\Exception $e) {
            Log::error('Error adding service to customer: ' . $e->getMessage());
            return back()
                ->with('error', 'Failed to add service. Please try again.')
                ->withInput();
        }
    }

    /**
     * Update a customer's service.
     */
    public function updateService(Request $request, Customer $customer, $serviceId)
    {
        $validated = $request->validate([
            'price' => 'required|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_active' => 'boolean',
            'notes' => 'nullable|string|max:500',
        ]);

        try {
            $customer->propServices()->updateExistingPivot($serviceId, [
                'Price' => $validated['price'],
                'StartDate' => $validated['start_date'],
                'EndDate' => $validated['end_date'] ?? null,
                'IsActive' => $validated['is_active'] ?? true,
                'Notes' => $validated['notes'] ?? null,
                'updated_at' => now(),
            ]);

            return redirect()->route('customers.services', $customer->CustomerID)
                ->with('success', 'Service updated successfully.');
                
        } catch (\Exception $e) {
            Log::error('Error updating customer service: ' . $e->getMessage());
            return back()
                ->with('error', 'Failed to update service. Please try again.')
                ->withInput();
        }
    }

    /**
     * Remove a service from the customer.
     */
    public function removeService(Customer $customer, $serviceId)
    {
        try {
            $customer->propServices()->detach($serviceId);
            
            return redirect()->route('customers.services', $customer->CustomerID)
                ->with('success', 'Service removed successfully.');
                
        } catch (\Exception $e) {
            Log::error('Error removing service from customer: ' . $e->getMessage());
            return back()
                ->with('error', 'Failed to remove service. Please try again.');
        }
    }
}
