<?php

namespace App\Http\Controllers;

use App\Models\Property;
use App\Models\Payment;
use App\Models\ProUnit;
use App\Models\User;
use App\Models\UnitAssignment;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index()
    {
        // Get the authenticated user's CompanyID
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            // Handle case where user doesn't have a CompanyID
            return redirect()->back()->with('error', 'No company associated with this account.');
        }
        
        // Get property statistics filtered by CompanyID
        $totalProperties = Property::where('CompanyID', $companyId)->count();
        
        // Get occupied properties count (active unit assignments) filtered by CompanyID
        $occupiedProperties = DB::table('unit_assignments')
            ->join('prounits', 'unit_assignments.unit_id', '=', 'prounits.UnitID')
            ->join('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('unit_assignments.status', 'active')
            ->where('property.CompanyID', $companyId)
            ->distinct('unit_assignments.customer_id')
            ->count('unit_assignments.customer_id');
            
        $occupancyRate = $totalProperties > 0 ? round(($occupiedProperties / $totalProperties) * 100) : 0;
        
        // Get current month's date range
        $currentMonthStart = now()->startOfMonth();
        $currentMonthEnd = now()->endOfMonth();
        
        // Get current month's payments filtered by CompanyID through customer relationship
        $monthlyPayments = Payment::whereBetween('payment.PaymentDate', [$currentMonthStart, $currentMonthEnd])
            ->join('customer', 'payment.CustomerID', '=', 'customer.CustomerID')
            ->join('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->join('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->sum('payment.TotalPayment');
            
        // Get current month's total rent from rentservicebill
        $totalRent = DB::table('rentservicebill')
            ->whereBetween('BillingDate', [$currentMonthStart, $currentMonthEnd])
            ->join('customer', 'rentservicebill.CustomerID', '=', 'customer.CustomerID')
            ->leftJoin('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->leftJoin('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->sum('TotalPayable');
            
        // Get last month's total rent for comparison
        $lastMonthRent = DB::table('rentservicebill')
            ->whereBetween('BillingDate', [
                now()->subMonth()->startOfMonth(),
                now()->subMonth()->endOfMonth()
            ])
            ->join('customer', 'rentservicebill.CustomerID', '=', 'customer.CustomerID')
            ->leftJoin('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->leftJoin('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->sum('TotalPayable');
            
        // Calculate rent percentage change
        $rentChange = $lastMonthRent > 0 
            ? round((($totalRent - $lastMonthRent) / $lastMonthRent) * 100)
            : ($totalRent > 0 ? 100 : 0);
            
        // Get current month's total utility bills (from bill table)
        $totalBills = DB::table('bill')
            ->whereBetween('BillingDate', [$currentMonthStart, $currentMonthEnd])
            ->where('BillType', '!=', 'RENT') // Exclude rent from utility bills
            ->join('customer', 'bill.CustomerID', '=', 'customer.CustomerID')
            ->leftJoin('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->leftJoin('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->sum('TotalBill');
            
        // Get last month's total bills for comparison
        $lastMonthTotalBills = DB::table('bill')
            ->whereBetween('BillingDate', [
                now()->subMonth()->startOfMonth(),
                now()->subMonth()->endOfMonth()
            ])
            ->join('customer', 'bill.CustomerID', '=', 'customer.CustomerID')
            ->leftJoin('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->leftJoin('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->sum('TotalBill');
            
        // Calculate total bills percentage change
        $totalBillsChange = $lastMonthTotalBills > 0 
            ? round((($totalBills - $lastMonthTotalBills) / $lastMonthTotalBills) * 100)
            : ($totalBills > 0 ? 100 : 0);
            
        // Get last month's payments for comparison
        $lastMonthStart = now()->subMonth()->startOfMonth();
        $lastMonthEnd = now()->subMonth()->endOfMonth();
        
        $lastMonthPayments = Payment::whereBetween('payment.PaymentDate', [$lastMonthStart, $lastMonthEnd])
            ->join('customer', 'payment.CustomerID', '=', 'customer.CustomerID')
            ->join('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->join('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->sum('payment.TotalPayment');
            
        // Calculate percentage change
        $paymentChange = $lastMonthPayments > 0 
            ? round((($monthlyPayments - $lastMonthPayments) / $lastMonthPayments) * 100)
            : ($monthlyPayments > 0 ? 100 : 0);
        
        // Get recent payments filtered by CompanyID through customer relationship
        $recentPayments = Payment::select('payment.*')
            ->join('customer', 'payment.CustomerID', '=', 'customer.CustomerID')
            ->join('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->join('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->with(['customer.unit.property'])
            ->orderBy('payment.PaymentDate', 'desc')
            ->take(5)
            ->get();
        
        // Prepare recent activities data
        $recentActivities = [
            [
                'title' => 'New Property Added',
                'description' => 'A new property has been added to the system',
                'time' => now()->subHours(2)->diffForHumans(),
                'icon' => 'fas fa-home',
                'color' => 'indigo'
            ],
            [
                'title' => 'Payment Received',
                'description' => 'New payment received from a tenant',
                'time' => now()->subHours(5)->diffForHumans(),
                'icon' => 'fas fa-money-bill-wave',
                'color' => 'green'
            ],
            [
                'title' => 'Maintenance Request',
                'description' => 'New maintenance request submitted',
                'time' => now()->subDays(1)->diffForHumans(),
                'icon' => 'fas fa-tools',
                'color' => 'yellow'
            ]
        ];

        // Get recent bills from database
        $recentBills = $this->getRecentBills();

        // Prepare stats array
        $stats = [
            'total_properties' => $totalProperties,
            'properties_change' => 5, // Example percentage change
            'occupied_properties' => $occupiedProperties,
            'occupancy_rate' => $occupancyRate,
            'occupancy_change' => 2, // Example percentage change
            'total_bills' => $totalBills,
            'bills_change' => $totalBillsChange,
            'total_rent' => $totalRent,
            'rent_change' => $rentChange,
            'monthly_payments' => $monthlyPayments,
            'payment_change' => $paymentChange,
            'recent_payments' => $recentPayments,
            'recent_activities' => $recentActivities,
            'recent_bills' => $recentBills,
            'recent_properties' => $this->getRecentProperties(),
        ];
        
        return view('dashboard', compact('stats'));
    }

    /**
     * Get recent bills with customer information
     *
     * @return array
     */
    protected function getRecentBills()
    {
        $companyId = $this->user()->CompanyID;
        
        return \App\Models\RentServiceBill::select('rentservicebill.*')
            ->join('customer', 'rentservicebill.CustomerID', '=', 'customer.CustomerID')
            ->leftJoin('prounits', 'customer.UnitID', '=', 'prounits.UnitID')
            ->leftJoin('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->with(['customer', 'unit'])
            ->orderBy('rentservicebill.BillingDate', 'desc')
            ->take(5)
            ->get()
            ->map(function($bill) {
                $billingDate = $bill->BillingDate ? \Carbon\Carbon::parse($bill->BillingDate) : null;
                $dueDate = $billingDate ? $billingDate->copy()->endOfMonth() : null;
                $isOverdue = $dueDate && now()->gt($dueDate) && $bill->BillStatus !== 'Paid';
                
                return [
                    'id' => $bill->BillID,
                    'customer_name' => $bill->customer->CustomerName ?? 'N/A',
                    'customer_phone' => $bill->customer->PhoneNumber ?? 'N/A',
                    'type' => 'Rent',
                    'amount' => (float) $bill->TotalPayable,
                    'due_date' => $dueDate ? $dueDate->format('M d, Y') : 'N/A',
                    'status' => $isOverdue ? 'Overdue' : ($bill->BillStatus === 'Paid' ? 'Paid' : 'Pending'),
                    'billing_date' => $billingDate ? $billingDate->format('M d, Y') : 'N/A',
                    'balance' => (float) ($bill->TotalPayable - $bill->TotalPaid),
                ];
            })
            ->toArray();
    }

    /**
     * Map bill type code to human-readable name
     *
     * @param string $typeCode
     * @return string
     */
    protected function getBillType($typeCode)
    {
        $types = [
            'RENT' => 'Rent',
            'WATER' => 'Water',
            'ELEC' => 'Electricity',
            'MTNC' => 'Maintenance',
            'OTHER' => 'Other',
        ];

        return $types[$typeCode] ?? $typeCode;
    }

    /**
     * Get recent properties with their units and tenant information
     *
     * @return array
     */
    protected function getRecentProperties()
    {
        $properties = [];
        
        // Get all active properties with their units
        $allProperties = Property::with(['units.unitAssignments.customer'])
            ->where('CompanyID', $this->user()->CompanyID)
            ->get();

        foreach ($allProperties as $property) {
            $totalUnits = $property->units->count();
            $occupiedUnits = $property->units->filter(function($unit) {
                return $unit->unitAssignments->where('status', 'active')->isNotEmpty();
            })->count();
            $totalRent = 0;
            $nextDueDate = null;
            $tenantName = 'Available';
            
            // Process each unit in the property
            foreach ($property->units as $unit) {
                $assignment = $unit->unitAssignments->where('status', 'active')->first();
                
                if ($assignment) {
                    $totalRent += $unit->MonthlyRent;
                    $tenantName = $assignment->customer->CustomerName ?? 'Occupied';
                    
                    // Calculate next due date (assuming rent is due on the 1st of each month)
                    $nextDueDate = now()->startOfMonth()->addMonth();
                }
            }
            
            // Calculate occupancy status
            $occupancyStatus = $occupiedUnits > 0 ? 'Occupied' : 'Vacant';
            $statusColor = $occupancyStatus === 'Occupied' ? 'green' : 'yellow';
            
            // Format property data
            $properties[] = [
                'PropID' => $property->PropID, // Add the property ID
                'name' => $property->PropName,
                'location' => $property->City,
                'status' => $occupancyStatus,
                'status_color' => $statusColor,
                'tenant' => $tenantName,
                'rent' => $totalRent,
                'due_date' => $nextDueDate ? $nextDueDate->format('M d, Y') : 'N/A',
                'units_total' => $totalUnits,
                'units_occupied' => $occupiedUnits,
                'occupancy_rate' => $totalUnits > 0 ? round(($occupiedUnits / $totalUnits) * 100) : 0
            ];
        }
        
        return $properties;
    }
}
