<?php

namespace App\Http\Controllers;

use App\Models\BulkInvoice;
use App\Models\Bill;
use App\Models\RentServiceBill;
use Carbon\Carbon;
use Illuminate\Http\Request;

class InvoiceTrackingController extends Controller
{
    public function dashboard(Request $request)
    {
        $companyId = session('CompanyID');
        $period = $request->input('period', Carbon::now()->format('Ym'));
        
        // Get counts by type
        $bulkInvoiceCount = BulkInvoice::where('CompanyID', $companyId)
            ->when($period, function($query) use ($period) {
                return $query->where('BillingPeriod', $period);
            })
            ->count();
            
        $billCount = Bill::whereHas('bulkInvoice', function($query) use ($companyId) {
            $query->where('CompanyID', $companyId);
        })
        ->when($period, function($query) use ($period) {
            return $query->where('BillingPeriod', $period);
        })
        ->count();
        
        $rentInvoiceCount = RentServiceBill::where('CompanyID', $companyId)
            ->when($period, function($query) use ($period) {
                return $query->where('BillingPeriod', $period);
            })
            ->count();
        
        // Get monthly trends for the last 12 months
        $monthlyTrends = $this->getMonthlyTrends($companyId);
        
        // Get today's generated invoices
        $todayInvoices = $this->getTodayInvoices($companyId);
        
        return view('tracking.invoices', [
            'bulkInvoiceCount' => $bulkInvoiceCount,
            'billCount' => $billCount,
            'rentInvoiceCount' => $rentInvoiceCount,
            'totalCount' => $bulkInvoiceCount + $billCount + $rentInvoiceCount,
            'monthlyTrends' => $monthlyTrends,
            'todayInvoices' => $todayInvoices,
            'selectedPeriod' => $period
        ]);
    }
    
    private function getMonthlyTrends($companyId)
    {
        $months = [];
        for ($i = 11; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i);
            $period = $month->format('Ym');
            
            $months[] = [
                'month' => $month->format('M Y'),
                'period' => $period,
                'bulk_invoices' => BulkInvoice::where('CompanyID', $companyId)
                    ->where('BillingPeriod', $period)
                    ->count(),
                'bills' => Bill::whereHas('bulkInvoice', function($query) use ($companyId, $period) {
                    $query->where('CompanyID', $companyId)->where('BillingPeriod', $period);
                })->count(),
                'rent_invoices' => RentServiceBill::where('CompanyID', $companyId)
                    ->where('BillingPeriod', $period)
                    ->count()
            ];
        }
        
        return $months;
    }
    
    private function getTodayInvoices($companyId)
    {
        $today = Carbon::today();
        
        return [
            'bulk_invoices' => BulkInvoice::where('CompanyID', $companyId)
                ->whereDate('InvoiceDate', $today)
                ->count(),
            'bills' => Bill::whereHas('bulkInvoice', function($query) use ($companyId) {
                $query->where('CompanyID', $companyId);
            })->whereDate('BillingDate', $today)->count(),
            'rent_invoices' => RentServiceBill::where('CompanyID', $companyId)
                ->whereDate('BillingDate', $today)
                ->count()
        ];
    }
}
