<?php

namespace App\Http\Controllers;

use Maatwebsite\Excel\Facades\Excel;
use App\Exports\MeterReadingExport;
use App\Exports\TMPMeterReadingExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\MeterReading;
use App\Models\TMP_MeterReading;
use App\Models\ProUnit;

class MeterReadingController extends Controller
{
    /**
     * Display a listing of the meter readings
     */
    public function index(Request $request)
    {
        // Get the authenticated user's CompanyID
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $query = MeterReading::with(['customer', 'unit.property'])
            ->whereHas('unit.property', function($q) use ($companyId) {
                $q->where('CompanyID', $companyId);
            })
            ->whereNotNull('Reading')
            ->whereNotNull('Consumption');

        // Get unique consumption types for the filter dropdown (filtered by company)
        $consumptionTypes = MeterReading::select('meterreading.ConsType')
            ->join('prounits', 'meterreading.UnitID', '=', 'prounits.UnitID')
            ->join('property', 'prounits.PropID', '=', 'property.PropID')
            ->where('property.CompanyID', $companyId)
            ->whereNotNull('meterreading.ConsType')
            ->whereNotNull('meterreading.Reading')
            ->whereNotNull('meterreading.Consumption')
            ->distinct()
            ->orderBy('meterreading.ConsType')
            ->pluck('meterreading.ConsType');

        // Get unique properties for the filter dropdown (filtered by company)
        $properties = \App\Models\Property::where('CompanyID', $companyId)
            ->orderBy('PropName')
            ->pluck('PropName', 'PropID');

        // Get unique billing periods for processed readings
        $processedBillingPeriods = MeterReading::select('BillingPeriod')
            ->distinct()
            ->orderBy('BillingPeriod', 'desc')
            ->pluck('BillingPeriod');

        // Get the latest billing period for default filtering
        $latestPeriod = $processedBillingPeriods->first();

        // Filter by billing period - use request value or default to latest period
        $selectedBillingPeriod = $request->filled('billing_period') ? $request->billing_period : $latestPeriod;
        if ($selectedBillingPeriod) {
            $query->where('BillingPeriod', $selectedBillingPeriod);
        }

        // Filter by consumption type if explicitly selected
        if ($request->filled('type')) {
            $query->where('ConsType', $request->type);
        }

        // Filter by property if explicitly selected
        if ($request->filled('property_id')) {
            $query->whereHas('unit', function($q) use ($request) {
                $q->where('PropID', $request->property_id);
            });
        }
        
        // Search by account number or customer name
        if ($request->filled('search')) {
            $searchTerm = '%' . $request->search . '%';
            $query->whereHas('customer', function($q) use ($searchTerm) {
                $q->where('AccountNo', 'LIKE', $searchTerm)
                  ->orWhere('CustomerName', 'LIKE', $searchTerm);
            });
        }

        // Get unique billing periods for unprocessed readings (from tmp_meterreading)
        $unprocessedBillingPeriods = \App\Models\TMP_MeterReading::select('BillingPeriod')
            ->distinct()
            ->orderBy('BillingPeriod', 'desc')
            ->pluck('BillingPeriod');
            
        // Get all unique billing periods for the dropdown
        $billingPeriods = $processedBillingPeriods->merge($unprocessedBillingPeriods)
            ->unique()
            ->sortDesc()
            ->values();

        $readings = $query->orderBy('ReadingDate', 'desc')->paginate(10);
            
        // Get the current billing period (either from request or the latest period)
        $currentBillingPeriod = $selectedBillingPeriod;
        
        // Get the active tab from the request or default to 'unprocessed'
        $activeTab = $request->input('active_tab', 'unprocessed');
            
        return view('meter-readings.index', [
            'readings' => $readings,
            'billingPeriods' => $billingPeriods,
            'processedBillingPeriods' => $processedBillingPeriods,
            'unprocessedBillingPeriods' => $unprocessedBillingPeriods,
            'consumptionTypes' => $consumptionTypes,
            'properties' => $properties,
            'currentBillingPeriod' => $currentBillingPeriod,
            'activeTab' => $activeTab
        ]);
    }

    /**
     * Show the form for creating a new meter reading
     */
    public function create()
    {
        return view('meter-readings.create');
    }

    /**
     * Store a newly created meter reading in storage
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'BillingPeriod' => 'required|string',
            'ConsType' => 'required|string|max:6',
            'CustomerID' => 'required|integer|exists:Customer,CustomerID',
            'UnitID' => 'required|integer|exists:ProUnit,UnitID',
            'Reading' => 'required|numeric',
            'ReadingDate' => 'required|date',
            'MeterNo' => 'nullable|string|max:50',
            'ReadingComments' => 'nullable|string',
            'CompanyID' => 'required|integer|exists:Company,CompanyID'
        ]);

        // Get previous reading
        $prevReading = TMP_MeterReading::where('UnitID', $validated['UnitID'])
            ->where('ConsType', $validated['ConsType'])
            ->orderBy('ReadingDate', 'desc')
            ->first();

        if ($prevReading) {
            $validated['PrvReading'] = $prevReading->Reading;
            $validated['PrvReadingDate'] = $prevReading->ReadingDate;
            $validated['PrvReadingID'] = $prevReading->MeterReadingID;
            $validated['Consumption'] = $validated['Reading'] - $prevReading->Reading;
        } else {
            $validated['PrvReading'] = 0;
            $validated['Consumption'] = $validated['Reading'];
        }

        $reading = MeterReading::create($validated);

        return redirect()->route('meter-readings.show', $reading->MeterReadingID)
            ->with('success', 'Meter reading added successfully.');
    }

    /**
     * Display the specified meter reading
     */
    public function show($id)
    {
        // Get the authenticated user's CompanyID
        $companyId = $this->user()->CompanyID;

        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        // First try to find in processed readings (meterreading table)
        $reading = MeterReading::with(['customer', 'unit.property'])
            ->whereHas('unit.property', function($q) use ($companyId) {
                $q->where('CompanyID', $companyId);
            })
            ->find($id);

        // If found in processed readings, check if there's missing data in the TMP table
        if ($reading) {
            $tmpReading = TMP_MeterReading::where('CustomerID', $reading->CustomerID)
                ->where('UnitID', $reading->UnitID)
                ->where('CompanyID', $reading->CompanyID)
                ->where('ConsType', $reading->ConsType)
                ->where('BillingPeriod', $reading->BillingPeriod)
                ->first();

            if ($tmpReading) {
                // Merge missing fields from TMP reading
                if (!$reading->ImagePath && $tmpReading->ImagePath) {
                    $reading->ImagePath = $tmpReading->ImagePath;
                }
                if (!$reading->Reading && $tmpReading->Reading) {
                    $reading->Reading = $tmpReading->Reading;
                }
                if (!$reading->Consumption && $tmpReading->Consumption) {
                    $reading->Consumption = $tmpReading->Consumption;
                }
                if (!$reading->PrvReading && $tmpReading->PrvReading) {
                    $reading->PrvReading = $tmpReading->PrvReading;
                }
                if (!$reading->MeterNo && $tmpReading->MeterNo) {
                    $reading->MeterNo = $tmpReading->MeterNo;
                }
                if (!$reading->ReadingComments && $tmpReading->ReadingComments) {
                    $reading->ReadingComments = $tmpReading->ReadingComments;
                }
            }
        }

        // If not found in processed readings, try unprocessed readings (tmp_meterreading table)
        if (!$reading) {
            $reading = TMP_MeterReading::with(['customer', 'unit.property'])
                ->whereHas('unit.property', function($q) use ($companyId) {
                    $q->where('CompanyID', $companyId);
                })
                ->find($id);

            // If still not found, return 404
            if (!$reading) {
                abort(404, 'Meter reading not found.');
            }

            // Mark as unprocessed for the view
            $reading->is_processed = false;
        } else {
            $reading->is_processed = true;
        }

        // Manually load previous and next readings with company filtering
        if ($reading->is_processed) {
            $reading->load([
                'previousReading' => function($query) use ($reading, $companyId) {
                    return $query->where('UnitID', $reading->UnitID)
                                ->where('ConsType', $reading->ConsType)
                                ->where('ReadingDate', '<', $reading->ReadingDate)
                                ->whereHas('unit.property', function($q) use ($companyId) {
                                    $q->where('CompanyID', $companyId);
                                })
                                ->orderBy('ReadingDate', 'desc')
                                ->limit(1);
                },
                'nextReadings' => function($query) use ($reading, $companyId) {
                    return $query->where('UnitID', $reading->UnitID)
                                ->where('ConsType', $reading->ConsType)
                                ->where('ReadingDate', '>', $reading->ReadingDate)
                                ->whereHas('unit.property', function($q) use ($companyId) {
                                    $q->where('CompanyID', $companyId);
                                })
                                ->orderBy('ReadingDate', 'asc');
                }
            ]);
        } else {
            // For unprocessed readings, load from tmp_meterreading table
            $reading->load([
                'previousReading' => function($query) use ($reading, $companyId) {
                    return $query->where('UnitID', $reading->UnitID)
                                ->where('ConsType', $reading->ConsType)
                                ->where('ReadingDate', '<', $reading->ReadingDate)
                                ->whereHas('unit.property', function($q) use ($companyId) {
                                    $q->where('CompanyID', $companyId);
                                })
                                ->orderBy('ReadingDate', 'desc')
                                ->limit(1);
                },
                'nextReadings' => function($query) use ($reading, $companyId) {
                    return $query->where('UnitID', $reading->UnitID)
                                ->where('ConsType', $reading->ConsType)
                                ->where('ReadingDate', '>', $reading->ReadingDate)
                                ->whereHas('unit.property', function($q) use ($companyId) {
                                    $q->where('CompanyID', $companyId);
                                })
                                ->orderBy('ReadingDate', 'asc');
                }
            ]);
        }

        return view('meter-readings.show', compact('reading'));
    }

    /**
     * Process pending meter readings from Android app
     */
    public function processPendingReadings()
    {
        try {
            DB::beginTransaction();
            
            // Get all pending readings
            $pendingReadings = TMP_MeterReading::all();
            $processed = 0;
            $errors = [];
            
            foreach ($pendingReadings as $tmpReading) {
                try {
                    // Convert to array and create in main table
                    $readingData = $tmpReading->toArray();
                    unset($readingData['TMP_ReadingID']); // Remove the temp ID
                    
                    // Get previous reading
                    $prevReading = MeterReading::where('UnitID', $tmpReading->UnitID)
                        ->where('ConsType', $tmpReading->ConsType)
                        ->orderBy('ReadingDate', 'desc')
                        ->first();
                    
                    if ($prevReading) {
                        $readingData['PrvReading'] = $prevReading->Reading;
                        $readingData['PrvReadingDate'] = $prevReading->ReadingDate;
                        $readingData['PrvReadingID'] = $prevReading->MeterReadingID;
                        $readingData['Consumption'] = $tmpReading->Reading - $prevReading->Reading;
                    } else {
                        $readingData['PrvReading'] = 0;
                        $readingData['Consumption'] = $tmpReading->Reading;
                    }
                    
                    // Create the reading
                    MeterReading::create($readingData);
                    
                    // Delete the processed reading
                    $tmpReading->delete();
                    $processed++;
                    
                } catch (\Exception $e) {
                    $errors[] = "Error processing reading ID {$tmpReading->TMP_ReadingID}: " . $e->getMessage();
                    Log::error("Error processing meter reading: " . $e->getMessage(), [
                        'reading' => $tmpReading->toArray(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => "Processed $processed readings successfully.",
                'errors' => $errors
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error in processPendingReadings: " . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to process readings: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * API endpoint for Android app to submit meter readings
     */
    public function apiStore(Request $request)
    {
        $validated = $request->validate([
            'BillingPeriod' => 'required|string',
            'ConsType' => 'required|string|max:6',
            'CustomerID' => 'required|integer|exists:Customer,CustomerID',
            'UnitID' => 'required|integer|exists:ProUnit,UnitID',
            'Reading' => 'required|numeric',
            'ReadingDate' => 'required|date',
            'MeterNo' => 'nullable|string|max:50',
            'CompanyID' => 'required|integer|exists:Company,CompanyID',
            'device_id' => 'required|string',
            'timestamp' => 'required|numeric',
            'location' => 'nullable|array',
            'location.latitude' => 'required_with:location|numeric',
            'location.longitude' => 'required_with:location|numeric',
            'location.accuracy' => 'nullable|numeric',
        ]);
        
        try {
            // Store in temporary table
            $readingData = [
                'BillingPeriod' => $validated['BillingPeriod'],
                'ConsType' => $validated['ConsType'],
                'CustomerID' => $validated['CustomerID'],
                'UnitID' => $validated['UnitID'],
                'Reading' => $validated['Reading'],
                'ReadingDate' => $validated['ReadingDate'],
                'MeterNo' => $validated['MeterNo'] ?? null,
                'CompanyID' => $validated['CompanyID'],
                'device_id' => $validated['device_id'],
                'location_data' => !empty($validated['location']) ? json_encode($validated['location']) : null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
            
            TMP_MeterReading::create($readingData);
            
            return response()->json([
                'success' => true,
                'message' => 'Meter reading received and queued for processing.'
            ]);
            
        } catch (\Exception $e) {
            Log::error("Error in meter reading API: " . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to save meter reading: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get previous meter readings for a unit and consumption type (for web interface)
     */
    public function getPreviousReadings(Request $request)
    {
        // Get the authenticated user's CompanyID
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return response()->json([
                'success' => false,
                'message' => 'No company associated with this account.'
            ], 403);
        }

        $request->validate([
            'unit_id' => [
                'required',
                'integer',
                'exists:ProUnit,UnitID,deleted_at,NULL',
                function ($attribute, $value, $fail) use ($companyId) {
                    $unit = \App\Models\ProUnit::find($value);
                    if ($unit && $unit->property->CompanyID != $companyId) {
                        $fail('The selected unit is invalid.');
                    }
                },
            ],
            'cons_type' => 'required|string|max:6',
            'limit' => 'nullable|integer|min:1|max:100',
        ]);

        $limit = $request->input('limit', 5);

        $readings = MeterReading::where('UnitID', $request->unit_id)
            ->where('ConsType', $request->cons_type)
            ->whereHas('unit.property', function($q) use ($companyId) {
                $q->where('CompanyID', $companyId);
            })
            ->orderBy('ReadingDate', 'desc')
            ->limit($limit)
            ->get()
            ->map(function($reading) {
                return [
                    'id' => $reading->MeterReadingID,
                    'reading' => (float)$reading->Reading,
                    'reading_date' => $reading->ReadingDate->format('Y-m-d H:i:s'),
                    'reading_date_formatted' => $reading->ReadingDate->format('M j, Y'),
                    'consumption' => $reading->Consumption ? (float)$reading->Consumption : null,
                    'base_of_billing' => $reading->BaseOfBilling,
                    'reading_comments' => $reading->ReadingComments,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $readings
        ]);
    }

    /**
     * API endpoint to get previous meter readings (for Android app)
     */
    public function apiGetPreviousReadings(Request $request)
    {
        // Get the authenticated user's CompanyID from the request or token
        $companyId = $request->user() ? $request->user()->CompanyID : $request->company_id;
        
        if (!$companyId) {
            return response()->json([
                'success' => false,
                'message' => 'No company associated with this account.'
            ], 403);
        }

        $request->validate([
            'unit_id' => [
                'required',
                'integer',
                'exists:ProUnit,UnitID,deleted_at,NULL',
                function ($attribute, $value, $fail) use ($companyId) {
                    $unit = \App\Models\ProUnit::find($value);
                    if ($unit && $unit->property->CompanyID != $companyId) {
                        $fail('The selected unit is invalid.');
                    }
                },
            ],
            'cons_type' => 'required|string|max:6',
            'limit' => 'nullable|integer|min:1|max:100',
            'company_id' => 'sometimes|required|integer|exists:Company,CompanyID',
        ]);

        $limit = $request->input('limit', 10);

        $readings = MeterReading::where('UnitID', $request->unit_id)
            ->where('ConsType', $request->cons_type)
            ->whereHas('unit.property', function($q) use ($companyId) {
                $q->where('CompanyID', $companyId);
            })
            ->orderBy('ReadingDate', 'desc')
            ->limit($limit)
            ->get()
            ->map(function($reading) {
                return [
                    'id' => $reading->MeterReadingID,
                    'reading' => (float)$reading->Reading,
                    'reading_date' => $reading->ReadingDate->format('Y-m-d H:i:s'),
                    'consumption' => $reading->Consumption ? (float)$reading->Consumption : null,
                    'meter_number' => $reading->MeterNo,
                    'comments' => $reading->ReadingComments,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $readings
        ]);
    }

    /**
     * Get unprocessed meter readings from tmp_meterreading table
     */
    public function getUnprocessedReadings(Request $request)
    {
        try {
            // Get the authenticated user's CompanyID
            $companyId = $this->user()->CompanyID;
            
            if (!$companyId) {
                return response()->json([
                    'success' => false,
                    'message' => 'No company associated with this account.'
                ], 403);
            }

            // Log the request parameters for debugging
            Log::info('Fetching unprocessed readings with params:', [
                'company_id' => $companyId,
                'type' => $request->type,
                'billing_period' => $request->billing_period
            ]);

            // Start building the query for tmp_meterreading with company filtering
            $query = TMP_MeterReading::query()
                ->whereHas('unit.property', function($q) use ($companyId) {
                    $q->where('CompanyID', $companyId);
                });

            // Eager load relationships with error handling and company filtering
            $query->with([
                'unit' => function($q) use ($companyId) {
                    $q->select('UnitID', 'UnitIdentity as UnitName', 'PropID')
                      ->whereHas('property', function($q) use ($companyId) {
                          $q->where('CompanyID', $companyId);
                      });
                }, 
                'customer' => function($q) use ($companyId) {
                    $q->select('CustomerID', 'CustomerName', 'AccountNo', 'UnitID')
                      ->whereHas('unit.property', function($q) use ($companyId) {
                          $q->where('CompanyID', $companyId);
                      });
                }
            ]);

            // Apply filters if provided
            if ($request->filled('type')) {
                $query->where('ConsType', $request->type);
            }

            if ($request->filled('property_id')) {
                $query->where('PropID', $request->property_id);
            }

            $billingPeriod = $request->filled('billing_period') 
                ? $request->billing_period 
                : now()->format('Y-m');
            
            $query->where('BillingPeriod', $billingPeriod);

            // Get the unit IDs and types that already have processed readings
            $processedReadings = MeterReading::query()
                ->where('BillingPeriod', $billingPeriod)
                ->whereNotNull('Reading')
                ->whereNotNull('Consumption')
                ->select('UnitID', 'ConsType')
                ->get()
                ->groupBy(['UnitID', 'ConsType']);

            // Only include readings that don't have a corresponding processed reading
            // for the same unit and consumption type
            $query->where(function($q) use ($processedReadings) {
                $q->whereNotIn('UnitID', $processedReadings->keys())
                  ->orWhere(function($q) use ($processedReadings) {
                      $q->whereIn('UnitID', $processedReadings->keys())
                        ->whereNotIn('ConsType', function($query) use ($processedReadings) {
                            $query->select('ConsType')
                                ->from('meterreading')
                                ->whereNotNull('Reading')
                                ->whereNotNull('Consumption')
                                ->whereColumn('UnitID', 'tmp_meterreading.UnitID')
                                ->whereColumn('BillingPeriod', 'tmp_meterreading.BillingPeriod');
                        });
                  });
            });

            // Get the results
            $perPage = $request->input('per_page', 10);
            $readings = $query->orderBy('ReadingDate', 'desc')->paginate($perPage);

            $formattedReadings = $readings->map(function($reading) {
                try {
                    // Get the unit name, using UnitIdentity as the primary source
                    $unitName = 'N/A';
                    if ($reading->unit) {
                        $unitName = $reading->unit->UnitName ?? 'N/A';
                    }
                    
                    // Safely format the reading date
                    $readingDate = 'N/A';
                    if ($reading->ReadingDate) {
                        try {
                            $readingDate = is_string($reading->ReadingDate) 
                                ? \Carbon\Carbon::parse($reading->ReadingDate)->format('Y-m-d H:i:s')
                                : $reading->ReadingDate->format('Y-m-d H:i:s');
                        } catch (\Exception $e) {
                            Log::warning('Failed to parse reading date: ' . $reading->ReadingDate);
                        }
                    }
                    
                    return [
                        'id' => $reading->MeterReadingID,
                        'meter_number' => $reading->MeterNo ?? 'N/A',
                        'unit' => $unitName,
                        'unit_id' => $reading->UnitID,
                        'customer' => $reading->customer ? $reading->customer->CustomerName : 'N/A',
                        'account_no' => $reading->customer ? $reading->customer->AccountNo : 'N/A',
                        'cons_type' => $reading->ConsType,
                        'reading' => (float)($reading->Reading ?? 0),
                        'previous_reading' => $reading->PrvReading ? (float)$reading->PrvReading : null,
                        'consumption' => $reading->Consumption ? (float)$reading->Consumption : null,
                        'reading_date' => $readingDate,
                        'billing_period' => $reading->BillingPeriod ?? 'N/A',
                        'comments' => $reading->ReadingComments ?? ''
                    ];
                } catch (\Exception $e) {
                    Log::error('Error formatting reading: ' . $e->getMessage());
                    Log::error('Reading data: ' . json_encode($reading));
                    return null;
                }
            })->filter();

            return response()->json([
                'success' => true,
                'data' => $formattedReadings->values(), // Reset array keys
                'pagination' => [
                    'current_page' => $readings->currentPage(),
                    'last_page' => $readings->lastPage(),
                    'per_page' => $readings->perPage(),
                    'total' => $readings->total(),
                    'from' => $readings->firstItem(),
                    'to' => $readings->lastItem(),
                    'has_pages' => $readings->hasPages(),
                ]
            ]);

        } catch (\Exception $e) {
            $errorMessage = 'Error fetching unprocessed readings: ' . $e->getMessage();
            Log::error($errorMessage);
            Log::error($e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => $errorMessage,
                'trace' => config('app.debug') ? $e->getTraceAsString() : null
            ], 500);
        }
    }

    /**
     * Update the specified meter reading in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            // Get the authenticated user's CompanyID
            $companyId = $this->user()->CompanyID;
            
            if (!$companyId) {
                return response()->json([
                    'success' => false,
                    'message' => 'No company associated with this account.'
                ], 403);
            }

            $validated = $request->validate([
                'current_reading' => 'required|numeric|min:0',
                'previous_reading' => 'nullable|numeric|min:0',
            ]);

            // Find the reading and verify it belongs to the user's company
            $reading = MeterReading::whereHas('unit.property', function($q) use ($companyId) {
                    $q->where('CompanyID', $companyId);
                })
                ->findOrFail($id);
            
            // Update the reading
            $reading->update([
                'Reading' => $validated['current_reading'],
                'PrvReading' => $validated['previous_reading'] ?? $reading->PrvReading,
                'Consumption' => $validated['current_reading'] - ($validated['previous_reading'] ?? $reading->PrvReading),
                'updated_by' => $this->user()->id,
                'updated_at' => now(),
            ]);

            // Update any subsequent readings that reference this one
            MeterReading::where('PrvReadingID', $reading->MeterReadingID)
                ->update([
                    'PrvReading' => $reading->Reading,
                    'Consumption' => DB::raw('Reading - ' . $reading->Reading),
                    'updated_by' =>  $this->user()->id,
                    'updated_at' => now(),
                ]);

            return response()->json([
                'success' => true,
                'message' => 'Meter reading updated successfully.',
                'data' => $reading->fresh()
            ]);

        } catch (\Exception $e) {
            Log::error('Error updating meter reading: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update meter reading: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export processed meter readings to Excel
     */
    public function exportProcessedReadings(Request $request)
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $billingPeriod = $request->input('billing_period');
        $type = $request->input('type');
        $propertyId = $request->input('property_id');
        $search = $request->input('search');

        $filename = 'processed-meter-readings';
        if ($billingPeriod) {
            $periodFormatted = \Carbon\Carbon::createFromFormat('Ym', $billingPeriod)->format('M-Y');
            $filename .= '-' . $periodFormatted;
        }
        $filename .= '-' . now()->format('Y-m-d') . '.xlsx';

        return Excel::download(
            new MeterReadingExport($billingPeriod, $type, $companyId, $search, $propertyId),
            $filename,
            \Maatwebsite\Excel\Excel::XLSX
        );
    }

    /**
     * Export unprocessed meter readings to Excel
     */
    public function exportUnprocessedReadings(Request $request)
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $billingPeriod = $request->input('billing_period');
        $type = $request->input('type');
        $propertyId = $request->input('property_id');
        $search = $request->input('search');

        $filename = 'unprocessed-meter-readings';
        if ($billingPeriod) {
            $periodFormatted = \Carbon\Carbon::createFromFormat('Ym', $billingPeriod)->format('M-Y');
            $filename .= '-' . $periodFormatted;
        }
        $filename .= '-' . now()->format('Y-m-d') . '.xlsx';

        return Excel::download(
            new TMPMeterReadingExport($billingPeriod, $type, $companyId, $search, $propertyId),
            $filename,
            \Maatwebsite\Excel\Excel::XLSX
        );
    }
}
