<?php

namespace App\Http\Controllers;

use App\Models\PropService;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PropServiceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $services = PropService::with(['company', 'property'])
            ->where('CompanyID', $companyId)
            ->orderBy('IDService', 'desc')
            ->paginate(15);
            
        return view('prop-services.index', compact('services'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
       
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        // Only show properties that belong to the user's company
        $properties = \App\Models\Property::where('CompanyID', $companyId)
            ->pluck('PropName', 'PropID');
            
        $propertyId = $request->query('property_id');
        
        // Get companies for the dropdown
        $companies = Company::where('CompanyID', $companyId)
            ->pluck('CompanyName', 'CompanyID');
            
        return view('prop-services.create', [
            'companyId' => $companyId,
            'companies' => $companies,
            'properties' => $properties,
            'propertyId' => $propertyId
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
       
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

       

        $validated = $request->validate([
            'ServiceName' => 'required|string|max:255',
            'Price' => 'required|numeric|min:0',
            'CompanyID' => 'required|exists:company,CompanyID',
            'PropID' => 'required|exists:property,PropID',
            'Description'=>'required|string|max:255'
            // Add validation for other fields as needed
        ]);

        try {
            DB::beginTransaction();
            
            $service = PropService::create([
                'ServiceName' => $validated['ServiceName'],
                'Price' => $validated['Price'],
                'CompanyID' => $validated['CompanyID'],
                'PropID' => $validated['PropID'],
                'Description' => $validated['Description']
            ]);
          
            DB::commit();
            
            return redirect()->route('prop-services.index')
                ->with('success', 'Service created successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating service: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            return back()->withInput()->with('error', 'Error creating service: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $propService = PropService::with(['company', 'property'])
            ->where('CompanyID', $companyId)
            ->findOrFail($id);
            
        return view('prop-services.show', compact('propService'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $service = PropService::where('CompanyID', $companyId)
            ->findOrFail($id);
            
        // Get properties that belong to the user's company
        $properties = \App\Models\Property::where('CompanyID', $companyId)
            ->pluck('PropName', 'PropID');
            
        return view('prop-services.edit', [
            'propService' => $service,
            'properties' => $properties,
            'companyId' => $companyId,
            'companies' => Company::where('CompanyID', $companyId)
                ->pluck('CompanyName', 'CompanyID')
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        $service = PropService::where('CompanyID', $companyId)
            ->findOrFail($id);

        $validated = $request->validate([
            'ServiceName' => 'required|string|max:100',
            'Description' => 'nullable|string',
            'Price' => 'required|numeric|min:0',
            'NoMonthPerYear' => 'required|integer|min:1|max:12',
            'CompanyID' => [
                'required',
                'exists:company,CompanyID',
                function ($attribute, $value, $fail) use ($companyId) {
                    if ($value != $companyId) {
                        $fail('The selected company is invalid.');
                    }
                },
            ],
            'PropID' => [
                'required',
                'exists:property,PropID',
                function ($attribute, $value, $fail) use ($companyId) {
                    if (!\App\Models\Property::where('PropID', $value)
                        ->where('CompanyID', $companyId)
                        ->exists()) {
                        $fail('The selected property is invalid.');
                    }
                },
            ],
        ]);

        try {
            DB::beginTransaction();
            
            $service->update($validated);
            
            DB::commit();
            
            // Redirect back to the property show page if PropID is provided
            if (!empty($validated['PropID'])) {
                return redirect()->route('properties.show', $validated['PropID'])
                    ->with('success', 'Service updated successfully.');
            }
            
            return redirect()->route('prop-services.index')
                ->with('success', 'Service updated successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating service: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            return back()->withInput()->with('error', 'Error updating service: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $companyId = $this->user()->CompanyID;
        
        if (!$companyId) {
            return redirect()->back()->with('error', 'No company associated with this account.');
        }

        try {
            $service = PropService::where('CompanyID', $companyId)
                ->findOrFail($id);
                
            // Check if service is being used by any units
            if ($service->units()->exists()) {
                return back()->with('error', 'Cannot delete service as it is assigned to one or more units.');
            }
            
            DB::beginTransaction();
            
            $propId = $service->PropID;
            $service->delete();
            
            DB::commit();
            
            // Redirect back to the property show page if PropID is provided
            if ($propId) {
                return redirect()->route('properties.show', $propId)
                    ->with('success', 'Service deleted successfully.');
            }
            
            return redirect()->route('prop-services.index')
                ->with('success', 'Service deleted successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting service: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            return back()->with('error', 'Error deleting service: ' . $e->getMessage());
        }
    }
}
