<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Property;
use App\Models\Customer;
use App\Models\UnitAssignment;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\FinancialReportExport;
use App\Exports\RentRollExport;
use App\Exports\BillSummaryExport;
use App\Models\Bill;
use App\Models\Company;
use App\Models\BulkInvoice;
use App\Models\BulkInvoiceLine;
use App\Models\BillLine;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Barryvdh\DomPDF\Facade\Pdf as PDF;
use Illuminate\Support\Facades\Log; 

class ReportController extends Controller
{
    public function index()
    {
        return view('reports.index');
    }


      public function generateBillSummaryPdf($customerId, Request $request)
    {
        \Log::info('Attempting to generate PDF for customer ID: ' . $customerId);
        
        try {
            // Find the customer
            $customer = \App\Models\Customer::find($customerId);
            
            if (!$customer) {
                \Log::error('Customer not found with ID: ' . $customerId);
                return response()->json(['error' => 'Customer not found'], 404);
            }
            
            // Get the bill based on customer ID and optional billing period
            $billQuery = $customer->bills();
            
            // If billing period is provided in the request, filter by it
            if ($request->has('billing_period') && !empty($request->billing_period)) {
                $billQuery->where('BillingPeriod', $request->billing_period);
                \Log::info('Filtering bill by billing period: ' . $request->billing_period);
            } else {
                // Default to the latest bill if no billing period is specified
                $billQuery->latest('BillingPeriod');
            }
            
            $bill = $billQuery->first();
                
            if (!$bill) {
                $errorMsg = $request->has('billing_period') 
                    ? 'No bill found for the specified billing period' 
                    : 'No bills found for this customer';
                \Log::error($errorMsg . ' - Customer ID: ' . $customerId);
                
                // Return a view with SweetAlert script instead of JSON response
                return response(
                    "<html>
                    <head>
                        <title>Bill Not Found</title>
                        <script src='https://cdn.jsdelivr.net/npm/sweetalert2@11'></script>
                    </head>
                    <body>
                        <script>
                            document.addEventListener('DOMContentLoaded', function() {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Bill Not Found',
                                    text: '" . addslashes($errorMsg) . "',
                                    confirmButtonColor: '#1e40af',
                                    confirmButtonText: 'OK',
                                    allowOutsideClick: false
                                }).then((result) => {
                                    // Close the tab/window after user clicks OK
                                    window.close();
                                });
                            });
                        </script>
                    </body>
                    </html>"
                )->header('Content-Type', 'text/html');
            }
            
            // Eager load relationships to prevent N+1 queries
            $bill->load(['billLines', 'customer']);
            
            $billLines = $bill->billLines;
            // Calculate total amount from bill lines using BLineAmount
            $totalBillAmount = $billLines->sum('BLineAmount');
            $fileName = 'bill-summary-' . ($customer->AccountNo ?? $customer->CustomerID) . '.pdf';
            
            // Get company information for the logged-in user
            $company = Company::find(auth()->user()->CompanyID);
            
            $pdf = PDF::loadView('reports.customer-bill-pdf', [
                'bill' => $bill,
                'customer' => $customer,
                'billLines' => $billLines,
                'totalBillAmount' => $totalBillAmount,
                'billingPeriod' => $bill->BillingPeriod,
                'company' => $company
            ]);

            if ($request->has('download')) {
                return $pdf->download($fileName);
            }

            if ($request->has('print')) {
                return $pdf->stream($fileName, ['Attachment' => false]);
            }

            // Default: inline display in iframe
            return $pdf->stream($fileName, ['Attachment' => false]);
        } catch (\Exception $e) {
            \Log::error('Error generating PDF: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            return response()->json([
                'error' => 'Error generating PDF',
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ], 500);
        }
    }



    public function financialReport(Request $request)
    {
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $propertyId = $request->input('property_id');
        $reportType = $request->input('report_type', 'summary');
        
        $query = Payment::with(['customer', 'bill'])
            ->whereBetween('PaymentDate', [$startDate, $endDate])
            ->where('TotalPayment', '>', 0);
            
        if ($propertyId) {
            $query->whereHas('customer.unit.property', function($q) use ($propertyId) {
                $q->where('PropID', $propertyId);
            });
        }
        
        $payments = $query->get();
        $properties = Property::all();
        
        $data = [
            'payments' => $payments,
            'properties' => $properties,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'selectedProperty' => $propertyId,
            'reportType' => $reportType
        ];
        
        return view('reports.financial', $data);
    }
    
   public function occupancyReport(Request $request)
    {
        $date = $request->input('date', now()->format('Y-m-d'));
        $propertyId = $request->input('property_id');
        
        $query = Property::with(['units.unitAssignments' => function($q) use ($date) {
            $q->where('status', 'active');
        }]);
        
        if ($propertyId) {
            $query->where('PropID', $propertyId);
        }
        
        $properties = $query->get()->map(function($property) use ($date) {
            $totalUnits = $property->units->count();
            $occupiedUnits = 0;
            
            foreach ($property->units as $unit) {
                $hasActiveAssignment = $unit->unitAssignments->contains(function($assignment) use ($date) {
                    // Convert dates to Y-m-d format for comparison
                    $startDate = \Carbon\Carbon::parse($assignment->start_date)->format('Y-m-d');
                    $endDate = $assignment->end_date ? \Carbon\Carbon::parse($assignment->end_date)->format('Y-m-d') : null;
                    $checkDate = \Carbon\Carbon::parse($date)->format('Y-m-d');
                    
                    return $assignment->status === 'active' && 
                        $startDate <= $checkDate && 
                        ($endDate === null || $endDate >= $checkDate);
                });
                
                if ($hasActiveAssignment) {
                    $occupiedUnits++;
                }
            }
            
            $property->total_units = $totalUnits;
            $property->occupied_units = $occupiedUnits;
            $property->vacant_units = max(0, $totalUnits - $occupiedUnits);
            $property->occupancy_rate = $totalUnits > 0 
                ? round(($occupiedUnits / $totalUnits) * 100, 2) 
                : 0;
            
            return $property;
        });
        
        $allProperties = Property::pluck('PropName', 'PropID');
        
        return view('reports.occupancy', [
            'properties' => $properties,
            'allProperties' => $allProperties,
            'reportDate' => $date,
            'selectedProperty' => $propertyId
        ]);
    }


    public function generateCustomerStatementPdf($customerId, Request $request)
    {
        Log::info('Generating customer statement for ID: ' . $customerId);
        
        try {
            // Get customer with their bills ordered by BillID descending (newest first)
            $customer = Customer::with(['bills' => function($query) {
                $query->orderBy('BillID', 'desc');
            }])->find($customerId);

            if (!$customer) {
                Log::error('Customer not found with ID: ' . $customerId);
                return response()->json(['error' => 'Customer not found'], 404);
            }

            // Get the billing period from request or use the most recent one
            $billingPeriod = $request->get('billing_period');
            if ($billingPeriod) {
                $bills = $customer->bills->where('BillingPeriod', $billingPeriod);
            } else {
                $bills = $customer->bills;
            }

            // Calculate running balance
            $runningBalance = 0;
            $statementLines = $bills->map(function($bill) use (&$runningBalance) {
                $balanceBF = $bill->BalanceBF;
                $totalBill = $bill->TotalBill;
                $totalPayments = $bill->TotalPayments;
                $balanceCF = $bill->BalanceCF;
                
                $runningBalance = $balanceCF; // Update running balance
                
                return [
                    'period' => \Carbon\Carbon::createFromFormat('Ym', $bill->BillingPeriod)->format('M Y'),
                    'bill_date' => $bill->BillingDate,
                    'prev_reading' => $bill->PrvReading,
                    'curr_reading' => $bill->CurReading,
                    'consumption' => $bill->ConsumptionBilled,
                    'balance_bf' => number_format($balanceBF, 2),
                    'payments' => number_format($totalPayments, 2),
                    'bill_amount' => number_format($totalBill, 2),
                    'balance_cf' => number_format($balanceCF, 2),
                ];
            })->values();

            // Get the company information for the currently logged-in user
            $company = Company::find($this->user()->CompanyID);
            
            $data = [
                'customer' => $customer,
                'statementDate' => now()->format('d-F-Y'),
                'statementLines' => $statementLines,
                'currentBalance' => $runningBalance,
                'company' => $company
            ];

            $pdf = PDF::loadView('reports.customer-statement-pdf', $data);
            
            if ($request->has('download')) {
                return $pdf->download("statement-{$customer->AccountNo}.pdf");
            }

            return $pdf->stream("statement-{$customer->AccountNo}.pdf");
        } catch (\Exception $e) {
            Log::error('Error generating statement: ' . $e->getMessage());
            return response()->json([
                'error' => 'Error generating statement',
                'message' => $e->getMessage()
            ], 500);
        }
    }
 
   public function billDisplay(Request $request)
    {
        $query = Customer::query();
            
        // Apply search filter if search term exists
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = '%' . $request->search . '%';
            $query->where(function($q) use ($searchTerm) {
                $q->where('CustomerName', 'like', $searchTerm)
                  ->orWhere('AccountNo', 'like', $searchTerm);
            });
        }
        
        $customers = $query->orderBy('CustomerID', 'asc')->paginate(20);
            
        $companies = \App\Models\Company::orderBy('CompanyName')->get();
        
        // Get unique bulk invoices with their periods
        $bulkInvoices = Bill::select('BulkInvoiceID', 'BillingPeriod')
            ->whereNotNull('BulkInvoiceID')
            ->orderBy('BulkInvoiceID', 'desc')
            ->distinct()
            ->get();
            
        // Pass customers instead of bills to the view
        $bills = $customers;

        return view('reports.bill-display', compact('bills', 'companies', 'bulkInvoices'));
    }
    
      public function billSummary()
    {
        
        // Get companies for dropdown
        $companies = Company::orderBy('CompanyName')->get();
        
        // Get unique bulk invoices with their periods, ordered by BillingPeriod in descending order
        $bulkInvoices = Bill::select('BulkInvoiceID', 'BillingPeriod')
            ->whereNotNull('BulkInvoiceID')
            ->orderBy('BillingPeriod', 'desc')
            ->orderBy('BulkInvoiceID', 'desc')
            ->distinct()
            ->get();
            
        // Get the latest bulk invoice ID
        $latestBulkInvoiceId = $bulkInvoices->isNotEmpty() ? $bulkInvoices->first()->BulkInvoiceID : null;

        $summary = Bill::select(
                'BillingPeriod as period_id',
                DB::raw('count(*) as total_bills'),
                DB::raw('sum(TotalBill) as total_amount'),
                DB::raw('sum(TotalPayments) as total_paid'),
                DB::raw('sum(BalanceCF) as total_balance')
            )
            ->groupBy('BillingPeriod')
            ->orderBy('BillingPeriod', 'desc')
            ->get();

        return view('reports.bill-summary', [
            'summary' => $summary,
            'companies' => $companies,
            'bulkInvoices' => $bulkInvoices,
            'selectedBulkInvoice' => $latestBulkInvoiceId
        ]);
    }

     public function billSummaryView(Request $request)
    {
        
        // Redirect to login if user doesn't have a CompanyID
        if (is_null($this->user()->CompanyID)) {
            return redirect()->route('login')->with('error', 'Your account is not associated with any company. Please contact administrator.');
        }
        
        try {
            // Get the authenticated user's company ID
            $companyId = $this->user()->CompanyID;
            $billingPeriod = $request->get('period');
            
            Log::info("Loading full screen summary - User Company ID: {$companyId}, Period: {$billingPeriod}");
            
            // Get the same data as the detailed endpoint
            $bills = DB::table('view_bill_print')
                ->when($billingPeriod, function($query) use ($billingPeriod) {
                    return $query->where('BillingPeriod', $billingPeriod);
                })
                ->orderBy('AccountNo', 'asc')
                ->get();
            
            // Format bill data
            $formattedBills = $bills->map(function($bill) {
                // Construct property location from Court, Floor, Suite, AddressLine
                $propertyLocation = trim(($bill->Court ?? '') . ' ' . ($bill->Floor ?? '') . ' ' . ($bill->Suite ?? '') . ' ' . ($bill->AddressLine ?? ''));
                $propertyLocation = $propertyLocation ?: 'N/A';
                
                return [
                    'BillID' => $bill->BillNo, // VIEW_BILL_PRINT uses BillNo instead of BillID
                    'AccountNo' => $bill->AccountNo,
                    'CustomerName' => $bill->CustomerName,
                    'PropertyLocation' => $propertyLocation,
                    'MeterNo' => $bill->EMeterNo ?? $bill->MeterNo,
                    'CurReading' => $bill->CurReading,
                    'PrvReading' => $bill->PrvReading,
                    'ConsumptionBilled' => $bill->ConsumptionBilled,
                    'TotalBill' => $bill->TotalBill,
                    'BillingDate' => $bill->BillingDate,
                ];
            });
            
            // Calculate totals
            $totals = [
                'amount_total' => $bills->sum('TotalBill'),
                'consumption_total' => $bills->sum('ConsumptionBilled'),
                'current_total' => $bills->sum('CurReading'),
                'previous_total' => $bills->sum('PrvReading'),
            ];
            
            // Format period for display
            $periodText = $billingPeriod ? $this->formatPeriod($billingPeriod) : 'All Periods';
            
            $data = [
                'bills' => $formattedBills,
                'totals' => $totals,
                'period' => $periodText,
                'billingDate' => now()->format('d-m-Y'),
                'totalBills' => $formattedBills->count()
            ];
            
            // Check if PDF format is requested
            if ($request->has('format') && $request->get('format') === 'pdf') {
                // Get company data
                $company = Company::find($companyId);
                
                // If company not found, try to get the first company as fallback
                if (!$company) {
                    $company = Company::first();
                    Log::warning('Using fallback company', [
                        'requested_company_id' => $companyId,
                        'fallback_company' => $company ? $company->id : 'No companies found in database'
                    ]);
                }
                
                // Log company data for debugging
                Log::info('Company Data:', [
                    'company_id' => $companyId,
                    'company' => $company ? [
                        'id' => $company->id,
                        'CompanyName' => $company->CompanyName,
                        'CompanyLogo' => $company->CompanyLogo,
                        'exists' => file_exists(public_path('logo/' . $company->CompanyLogo)) ? 'Yes' : 'No'
                    ] : 'No company found',
                    'logo_path' => $company && $company->CompanyLogo ? 'logo/' . $company->CompanyLogo : 'No logo specified',
                    'logo_exists' => $company && $company->CompanyLogo ? 
                        (file_exists(public_path('logo/' . $company->CompanyLogo)) ? 'Yes' : 'No') : 'N/A'
                ]);
                
                $data['company'] = $company;
                
                $pdf = PDF::loadView('reports.bill-summary-pdf', $data);
                
                // Set paper size and orientation
                $pdf->setPaper('A4', 'portrait');
                
                // Format the period for filename (replace spaces and special characters)
                $formattedPeriod = strtolower(str_replace(' ', '-', $periodText));
                
                // Set download or view in browser
                if ($request->has('download') && $request->get('download') === '1') {
                    $filename = 'bill-summary-' . $formattedPeriod . '.pdf';
                    return $pdf->download($filename);
                }
                
                return $pdf->stream('bill-summary-' . $formattedPeriod . '.pdf');
            }
            
            return view('reports.bill-summary-view', $data);
            
        } catch (\Exception $e) {
            Log::error('Error in billSummaryView: ' . $e->getMessage());
            return back()->with('error', 'Error loading report: ' . $e->getMessage());
        }
    }

    
    private function formatPeriod($period)
    {
        if ($period && strlen($period) === 6) {
            $year = substr($period, 0, 4);
            $month = substr($period, 4, 2);
            $monthNames = ['January', 'February', 'March', 'April', 'May', 'June', 
                          'July', 'August', 'September', 'October', 'November', 'December'];
            return $monthNames[intval($month) - 1] . ' ' . $year;
        }
        return $period;
    }

    public function billSummaryReport(Request $request)
    {
      
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $propertyId = $request->input('property_id');
        
        // Load bills with customer and customer's unit relationship
        $query = Bill::with([
            'customer',
            'customer.unit' => function($q) {
                $q->select('UnitID', 'UnitIdentity', 'PropID');
            },
            'customer.unit.property' => function($q) {
                $q->select('PropID', 'PropName');
            }
        ])
        ->whereBetween('BillingDate', [$startDate, $endDate])
        ->orderBy('BillingDate', 'desc');
            
        if ($propertyId) {
            $query->whereHas('customer.unit', function($q) use ($propertyId) {
                $q->where('PropID', $propertyId);
            });
        }
        
        $bills = $query->paginate(25);
        $properties = Property::select('PropID', 'PropName')->get();
        
        return view('reports.bill-summary', [
            'bills' => $bills,
            'properties' => $properties,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'selectedProperty' => $propertyId
        ]);
    }


    public function varianceReport( Request $request){

        // Get the billing period from the request or use a default (YYYYMM format)
        $period = $request->query('period', date('Ym'));
        $bulkInvoiceId = $request->query('bulk_invoice_id');

        // Convert period to YYYYMM format for database queries
        $periodPrefix = $period;
        if (preg_match('/^[A-Za-z]+\s+\d{4}$/', $period)) {
            // Handle "January 2026" format
            try {
                $date = \Carbon\Carbon::createFromFormat('F Y', $period);
                $periodPrefix = $date->format('Ym');
            } catch (\Exception $e) {
                Log::error('Invalid period format: ' . $period . ', using current month');
                $periodPrefix = date('Ym');
            }
        } elseif (!preg_match('/^\d{6}$/', $period)) {
            // If not YYYYMM format, try to convert from other formats
            try {
                $periodPrefix = date('Ym', strtotime($period));
            } catch (\Exception $e) {
                Log::error('Could not convert period: ' . $period . ', using current month');
                $periodPrefix = date('Ym');
            }
        }

        // Get company information
        $companyId = $this->user()->CompanyID ?? session('CompanyID');
        $company = Company::find($companyId);

        if (!$company) {
            // Fallback to first company if user doesn't have a company
            $company = Company::first();
            Log::warning('User does not have a company assigned. Using first company as fallback.');
        }

        // Query the view_bill_summ2 for the specified period
        $viewData = DB::select("SELECT * FROM view_bill_summ2 WHERE BillingPeriod = ?", [$periodPrefix]);

        // Initialize arrays for the blade template
        $lineItems = [];
        $kplcBill = [];
        $totalBilling = [];
        $kwhValues = [
            'kplc_kwh' => 0,
            'total_billing_kwh' => 0,
            'variance_kwh' => 0
        ];

        $invoiceNo = null;

        // Process the view data
        foreach ($viewData as $row) {
            $customerName = $row->CustomerName;
            $bLineDesc = $row->BLineDesc;
            $amount = (float)$row->Amount;
            $lineType = trim($row->LineType);

            // Extract line ID from BLineDesc (e.g., "01. Fixed Charge" -> "01")
            $lineId = '';
            if (preg_match('/^(\d+)\./', $bLineDesc, $matches)) {
                $lineId = $matches[1];
            }

            // Initialize line item if not exists
            if (!isset($lineItems[$lineId]) && $lineId !== '') {
                $lineItems[$lineId] = [
                    'id' => $lineId,
                    'description' => $bLineDesc
                ];
            }

            // Process based on customer type
            if (strpos($customerName, 'Total Billing') === 0) {
                // Total Billing data (negative amounts)
                if ($lineType === 'Kshs.') {
                    if (!isset($totalBilling[$lineId])) {
                        $totalBilling[$lineId] = 0;
                    }
                    $totalBilling[$lineId] += abs($amount); // Convert negative to positive for display
                } elseif ($lineType === 'Kwh') {
                    $kwhValues['total_billing_kwh'] += abs($amount);
                }
            } elseif (strpos($customerName, 'KPLC Bill') === 0) {
                // KPLC Bill data (positive amounts)
                if ($lineType === 'Kshs.') {
                    if (!isset($kplcBill[$lineId])) {
                        $kplcBill[$lineId] = 0;
                    }
                    $kplcBill[$lineId] += $amount;
                } elseif ($lineType === 'Kwh') {
                    $kwhValues['kplc_kwh'] += $amount;
                }

                // Extract invoice number
                if (!$invoiceNo && preg_match('/KPLC Bill\s+(.+)/', $customerName, $matches)) {
                    $invoiceNo = trim($matches[1]);
                }
            }
        }

        // Calculate totals
        $totalBilling['total'] = array_sum(array_filter($totalBilling, function($key) {
            return $key !== 'total';
        }, ARRAY_FILTER_USE_KEY));

        $kplcBill['total'] = array_sum(array_filter($kplcBill, function($key) {
            return $key !== 'total';
        }, ARRAY_FILTER_USE_KEY));

        // Calculate variance
        $variance = [];
        foreach ($lineItems as $lineId => $lineItem) {
            $variance[$lineId] = ($kplcBill[$lineId] ?? 0) - ($totalBilling[$lineId] ?? 0);
        }
        $variance['total'] = ($kplcBill['total'] ?? 0) - ($totalBilling['total'] ?? 0);

        // Calculate KWH variance
        $kwhValues['variance_kwh'] = $kwhValues['kplc_kwh'] - $kwhValues['total_billing_kwh'];

        // Log the data for debugging
        Log::info('Variance Report Data', [
            'period' => $period,
            'bulkInvoiceId' => $bulkInvoiceId,
            'companyId' => $company ? $company->CompanyID : null,
            'viewRecordsCount' => count($viewData),
            'lineItemsCount' => count($lineItems),
            'invoiceNo' => $invoiceNo
        ]);

        return view('reports.variance-report', [
            'period' => $period,
            'company' => $company,
            'kplcBill' => $kplcBill,
            'totalBilling' => $totalBilling,
            'variance' => $variance,
            'bulkInvoiceId' => $bulkInvoiceId,
            'invoiceNo' => $invoiceNo,
            'kwhValues' => $kwhValues,
            'lineItems' => $lineItems,
        ]);
    }

    public function detailReport(Request $request)
{
    // Get the billing period from the request or use a default (in YYYYMM format)
    $period = $request->query('period', date('Ym'));
    
    // Format the period for display
    $displayPeriod = $this->formatPeriod($period);
    
    // Get company information
    $companyId = $this->user()->CompanyID ?? session('CompanyID');
    $company = Company::find($companyId);
    
    if (!$company) {
        $company = Company::first();
    }
    
    // Query view_bill_summ1 for the specified period
    $viewData = DB::select("SELECT * FROM view_bill_summ1 WHERE BillingPeriod = ?", [$period]);
    
    // Group data by customer
    $groupedCustomers = [];
    foreach ($viewData as $row) {
        $customerName = $row->CustomerName;
        
        if (!isset($groupedCustomers[$customerName])) {
            $groupedCustomers[$customerName] = [
                'customer_name' => $customerName,
                'total' => 0,
                'lines' => []
            ];
        }
        
        // Extract line ID from BLineDesc (e.g., "01. Fixed Charge" -> "01")
        $lineId = '';
        if (preg_match('/^(\d+)\./', $row->BLineDesc, $matches)) {
            $lineId = $matches[1];
        }
        
        // Initialize line item if not exists
        if (!isset($groupedCustomers[$customerName]['lines'][$row->BLineDesc])) {
            $groupedCustomers[$customerName]['lines'][$row->BLineDesc] = [
                'amount' => 0,
                'description' => $row->BLineDesc,
                'BLineID' => $lineId
            ];
        }
        
        // Sum amounts for the same description
        $groupedCustomers[$customerName]['lines'][$row->BLineDesc]['amount'] += (float)$row->Amount;
        $groupedCustomers[$customerName]['total'] += (float)$row->Amount;
    }
    
    // Sort customers by name in ascending order
    ksort($groupedCustomers);
    
    // Convert to the format expected by the view
    $customers = collect($groupedCustomers)->map(function($customerData) {
        $result = [
            'customer_name' => $customerData['customer_name'],
            'total' => $customerData['total']
        ];
        
        // Add line items in the expected format
        foreach ($customerData['lines'] as $line) {
            $key = 'line_' . $line['BLineID'];
            $result[$key] = [
                'amount' => $line['amount'],
                'description' => $line['description'],
                'BLineID' => $line['BLineID']
            ];
        }
        
        // Debug: Log the result structure
        Log::info('Customer result structure for ' . $customerData['customer_name'], [
            'customer_name' => $customerData['customer_name'],
            'total' => $customerData['total'],
            'line_keys' => array_filter(array_keys($result), function($key) {
                return strpos($key, 'line_') === 0;
            }),
            'sample_line_data' => isset($result['line_01']) ? $result['line_01'] : 'No line_01 found'
        ]);
        
        return $result;
    });
    
    // Get bill line descriptions (same as current implementation)
    $billLineDescriptions = [];
    try {
        // Temporarily force use of defaults to test
        throw new \Exception('Testing with defaults');
        $billLineDescriptions = DB::table('tariffline')
            ->orderBy('TLineID')
            ->pluck('TLineName')
            ->toArray();
    } catch (\Exception $e) {
        // Use default descriptions that match view_bill_summ1
        $billLineDescriptions = [
            '01. Fixed Charge',
            '02. Consumption',
            '03. Max Demand KVA',
            '04. Fuel Energy Cost',
            '05. Forex Exchange Adj.',
            '06. Inflation Adj.',
            '07. ERC Levy',
            '08. REP Levy',
            '09. WARMA LEVY',
            '10. VAT @ 16%'
        ];
    }
    
    return view('reports.detail-report', [
        'period' => $displayPeriod,
        'company' => $company,
        'customers' => $customers,
        'billLineDescriptions' => $billLineDescriptions
    ]);
}

 public function detailReport2(Request $request, $bulkInvoiceId = null)
        {
            // Get the bulk invoice ID from either route parameter or query string
            $bulkInvoiceId = $bulkInvoiceId ?? $request->query('bulk_invoice_id');
            
            if (!$bulkInvoiceId) {
                return redirect()->back()->with('error', 'Bulk Invoice ID is required');
            }
            
            // Check if this is a PDF export request
            if ($request->has('export') && $request->export === 'pdf') {
                return $this->exportPdf($request, $bulkInvoiceId);
            }
            
            // Store the bulk invoice ID in the session for pagination
            $request->session()->put('current_bulk_invoice_id', $bulkInvoiceId);

            // Get company information
            $company = Company::find($this->user()->CompanyID);
            if (!$company) {
                return redirect()->back()->with('error', 'Company not found');
            }

            // Query view_bill_summ1b for the bulk invoice
            $viewData = DB::select("SELECT * FROM view_bill_summ1b WHERE BillNo IN (
                SELECT BillID FROM bill WHERE BulkInvoiceID = ?
            )", [$bulkInvoiceId]);

            // Group data by customer (which already includes address)
            $groupedCustomers = [];
            $totalAmount = 0;

            foreach ($viewData as $row) {
                $customerName = $row->CustomerName; // Already "Address - CustomerName" format

                if (!isset($groupedCustomers[$customerName])) {
                    $groupedCustomers[$customerName] = [
                        'name' => $customerName,
                        'total' => 0,
                        'lines' => []
                    ];
                }

                // Extract line ID from BLineDesc (e.g., "01. Fixed Charge" -> "01")
                $lineId = '';
                if (preg_match('/^(\d+)\./', $row->BLineDesc, $matches)) {
                    $lineId = $matches[1];
                }

                // Initialize line item if not exists
                if (!isset($groupedCustomers[$customerName]['lines'][$lineId])) {
                    $groupedCustomers[$customerName]['lines'][$lineId] = [
                        'amount' => 0,
                        'description' => $row->BLineDesc
                    ];
                }

                // Sum amounts for the same line ID
                $groupedCustomers[$customerName]['lines'][$lineId]['amount'] += (float)$row->Amount;
                $groupedCustomers[$customerName]['total'] += (float)$row->Amount;
                $totalAmount += (float)$row->Amount;
            }

            // Sort customers by name
            ksort($groupedCustomers);

            // Create line descriptions array from the grouped data
            $lineDescriptions = [];
            foreach ($groupedCustomers as $customer) {
                foreach ($customer['lines'] as $lineId => $line) {
                    if (!isset($lineDescriptions[$lineId])) {
                        $lineDescriptions[$lineId] = $line['description'];
                    }
                }
            }
            ksort($lineDescriptions);

            // Calculate column totals
            $columnTotals = [];
            foreach ($lineDescriptions as $lineId => $description) {
                $columnTotals[$lineId] = 0;
                foreach ($groupedCustomers as $customer) {
                    if (isset($customer['lines'][$lineId])) {
                        $columnTotals[$lineId] += $customer['lines'][$lineId]['amount'];
                    }
                }
            }

            // Get billing period from one of the bills
            $billingPeriod = '';
            if (!empty($viewData)) {
                $billingPeriod = $viewData[0]->BillingPeriod;
            }

            // Paginate the results
            $perPage = 10;
            $currentPage = \Illuminate\Pagination\Paginator::resolveCurrentPage() ?: 1;
            $customerArray = array_values($groupedCustomers);
            $currentItems = array_slice($customerArray, ($currentPage - 1) * $perPage, $perPage, true);
            $paginatedCustomerTotals = new \Illuminate\Pagination\LengthAwarePaginator(
                $currentItems,
                count($customerArray),
                $perPage,
                $currentPage,
                [
                    'path' => route('reports.detail-report-2', ['bulk_invoice_id' => $bulkInvoiceId]),
                    'query' => $request->query()
                ]
            );

            $viewData = [
                'customerTotals' => $paginatedCustomerTotals,
                'lineDescriptions' => $lineDescriptions,
                'totalAmount' => $totalAmount,
                'billingPeriod' => $billingPeriod,
                'bulkInvoiceId' => $bulkInvoiceId,
                'columnTotals' => $columnTotals,
                'isPdf' => false
            ];

            return view('reports.detail-report-2', $viewData);
        }


   
public function vatSummary(Request $request)
{
    try {
        // Get the authenticated user's company ID
        $companyId = $this->user()->CompanyID;
        $billingPeriod = $request->get('period');
        $bulkInvoiceId = $request->get('bulk_invoice_id');
        
        if (!$bulkInvoiceId) {
            return back()->with('error', 'Bulk invoice ID is required');
        }
        
        // Get company information
        $company = Company::find($companyId);
        
        // Get all bills with VAT breakdown from view_bill_print2 for the selected bulk invoice
        $bills = DB::table('view_bill_print2')
            ->where('BulkInvoiceID', $bulkInvoiceId)
            ->orderBy('AccountNo')
            ->orderBy('BillNo')
            ->get();
            
        if ($bills->isEmpty()) {
            return back()->with('error', 'No bills found for the selected bulk invoice');
        }
        
        // Initialize summary
        $vatSummary = [
            'total_vatable' => 0,
            'total_non_vatable' => 0,
            'total_vat' => 0,
            'total_invoice' => 0,
            'bill_count' => $bills->unique('BillNo')->count(),
            'billing_period' => $bills->first()->BillingPeriod ?? '',
            'billing_date' => now()->format('d-m-Y'),
            'bulk_invoice_id' => $bulkInvoiceId
        ];
        
        // Group bills by customer (using AccountNo as customer identifier)
        $customerVatData = [];
        $groupedBills = $bills->groupBy('AccountNo');
        
        foreach ($groupedBills as $accountNo => $customerBills) {
            $customerData = $customerBills->first();
            $customerTotal = $customerBills->sum('TotalBill');
            $customerVatable = $customerBills->sum('VATABLE');
            $customerNonVatable = $customerBills->sum('NONVATABLE');
            $customerVat = $customerBills->sum('VATAmount');
            $totalConsumption = $customerBills->sum('ConsumptionBilled');

            // Add customer to VAT data
            $customerVatData[] = [
                'account_no' => $customerData->AccountNo,
                'customer_name' => $customerData->CustomerName,
                'address1' => $customerData->Address1 ?? 'N/A',
                'consumption' => round($totalConsumption, 2),
                'tax_pin' => 'N/A', // Tax PIN field not available in customer table
                'vatable_amount' => round($customerVatable, 2),
                'non_vatable_amount' => round($customerNonVatable, 2),
                'vat_amount' => round($customerVat, 2),
                'total_amount' => round($customerTotal, 2)
            ];
            
            // Update summary totals
            $vatSummary['total_vatable'] += $customerVatable;
            $vatSummary['total_non_vatable'] += $customerNonVatable;
            $vatSummary['total_vat'] += $customerVat;
            $vatSummary['total_invoice'] += $customerTotal;
        }
        
        // Format summary numbers to 2 decimal places
        $vatSummary['total_vatable'] = round($vatSummary['total_vatable'], 2);
        $vatSummary['total_non_vatable'] = round($vatSummary['total_non_vatable'] ?? 0, 2);
        $vatSummary['total_vat'] = round($vatSummary['total_vat'], 2);
        $vatSummary['total_invoice'] = round($vatSummary['total_invoice'], 2);
        
        // Format billing period for display
        $vatSummary['period_text'] = $this->formatPeriod($vatSummary['billing_period']);
        
        // Check if PDF format is requested
        if ($request->has('format') && $request->get('format') === 'pdf') {
            $pdf = PDF::loadView('reports.vat-summary-pdf', [
                'vatSummary' => $vatSummary,
                'customerVatData' => $customerVatData,
                'company' => $company
            ]);
            
            $pdf->setPaper('A4', 'landscape'); // Changed to landscape for better readability
            $formattedPeriod = strtolower(str_replace(' ', '-', $vatSummary['period_text']));
            
            if ($request->has('download') && $request->get('download') === '1') {
                return $pdf->download("vat-summary-{$formattedPeriod}.pdf");
            }
            
            return $pdf->stream("vat-summary-{$formattedPeriod}.pdf");
        }
        
        // For HTML view
        return view('reports.vat-summary', [
            'vatSummary' => $vatSummary,
            'customerVatData' => $customerVatData,
            'company' => $company
        ]);
        
    } catch (\Exception $e) {
        Log::error('Error generating VAT summary: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
        return back()->with('error', 'Error generating VAT summary: ' . $e->getMessage());
    }
}


    /**
     * Display bills for printing using the view_bill_print_ex view
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response|\Barryvdh\DomPDF\PDF|\Illuminate\View\View
     */
    public function printBills(Request $request)
    {
        $request->validate([
            'bulk_invoice_id' => 'required|exists:bulkinvoice,BulkInvoiceID',
            'format' => 'sometimes|in:html,pdf',
            'download' => 'sometimes|in:0,1'
        ]);

        $bulkInvoiceId = $request->input('bulk_invoice_id');
        $format = $request->input('format', 'html');
        $download = $request->boolean('download', false);

        // Get bills from the view
        $bills = DB::table('view_bill_print_ex')
            ->where('BulkInvoiceID', $bulkInvoiceId)
            ->orderBy('CourtID')
            ->orderBy('ShopNo')
            ->get();

        // Get company information for the logged-in user
        $company = null;
        if ($this->user()->CompanyID) {
            $company = Company::find($this->user()->CompanyID);
        } 
        // Fallback to first company if not found
        if (!$company) {
            $company = Company::first();
        }

        // Group bills by court for the view
        $groupedBills = $bills->groupBy('CourtID');

        // Get billing period from the first bill (all bills in the same bulk invoice should have the same period)
        $billingPeriod = $bills->first() ? $bills->first()->BillingPeriod : '';

        // Prepare data for all bills (each bill will have its own customer, bill lines, and totals)
        $billData = [];
        foreach ($bills as $bill) {
            // Create customer object for this bill
            $customer = (object) [
                'TenantName' => $bill->TenantName,
                'customer_name' => $bill->CustomerName,
                'TelNo' => $bill->Suite, // TelNo is aliased as Suite in the view
                'Address1' => $bill->Court, // Address1 is aliased as Court
                'Address2' => $bill->Floor, // Address2 is aliased as Floor
                'AddressLine' => $bill->AddressLine,
                'AccountNo' => $bill->AccountNo,
                'EMeterNo' => $bill->EMeterNo,
            ];

            // Get bill lines for this specific bill
            $billLines = DB::table('billline')
                ->where('BillID', $bill->BillNo) // BillID is aliased as BillNo in view
                ->orderBy('BLineID')
                ->get();

            // Calculate total for this bill
            $totalBillAmount = $bill->TotalBill ?? 0;

            $billData[] = [
                'bill' => $bill,
                'customer' => $customer,
                'billLines' => $billLines,
                'totalBillAmount' => $totalBillAmount,
                'totalAmount' => $totalBillAmount, // Same for now
            ];
        }

        $data = [
            'bills' => $bills,
            'billData' => $billData, // Array of data for each individual bill
            'groupedBills' => $groupedBills,
            'company' => $company,
            'billingPeriod' => $billingPeriod,
            'bulkInvoiceId' => $bulkInvoiceId,
        ];

        if ($format === 'pdf') {
            // For PDF, use the layout version
            $pdf = PDF::loadView('reports.print-bills-pdf-layout', $data)
                ->setPaper('a4', 'portrait')
                ->setOption('isHtml5ParserEnabled', true)
                ->setOption('isRemoteEnabled', true);
            
            if ($download) {
                return $pdf->download('bills-' . $bulkInvoiceId . '.pdf');
            }
            
            return $pdf->stream('bills-' . $bulkInvoiceId . '.pdf');
        }

        // Generate and stream PDF in new tab instead of downloading
        $pdf = PDF::loadView('reports.print-bills', $data)
            ->setPaper('a4', 'portrait')
            ->setOption('isHtml5ParserEnabled', true)
            ->setOption('isRemoteEnabled', true);

        return $pdf->stream('bills-' . $bulkInvoiceId . '.pdf')
            ->withHeaders([
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="bills-' . $bulkInvoiceId . '.pdf"'
            ]);
    }

private function exportPdf(Request $request, $bulkInvoiceId)
    {
        // Get company information
        $company = Company::find($this->user()->CompanyID);
        if (!$company) {
            return redirect()->back()->with('error', 'Company not found');
        }

        // Get the billing period from the bulk invoice
        $bulkInvoice = Bill::where('BulkInvoiceID', $bulkInvoiceId)->first();
        if (!$bulkInvoice) {
            return redirect()->back()->with('error', 'Bulk Invoice not found');
        }

        $billingPeriod = $bulkInvoice->BillingPeriod;
        $displayPeriod = \Carbon\Carbon::createFromFormat('Ym', $billingPeriod)->format('F Y');

        // Get all bills for this bulk invoice with their customers and lines
        $bills = Bill::with(['customer', 'billLines' => function($query) {
                $query->orderBy('BLineID');
            }])
            ->where('BulkInvoiceID', $bulkInvoiceId)
            ->join('customer', 'bill.CustomerID', '=', 'customer.CustomerID')
            ->orderBy('customer.Address1')
            ->orderBy('customer.CustomerName')
            ->select('bill.*')
            ->get();

        // Group bills by customer and sum their totals
        $customerTotals = [];
        $totalAmount = 0;
        $lineDescriptions = [];

        foreach ($bills as $bill) {
            $customerId = $bill->CustomerID;

            // Group by Address1 and CustomerName
            $address = $bill->customer ? trim($bill->customer->Address1) : 'UNKNOWN';
            $customerName = $bill->customer ? trim($bill->customer->CustomerName) : 'UNKNOWN';
            $groupKey = $address . '|' . $customerName; // Composite key
            $displayName = $address . ' - ' . $customerName;

            if (!isset($customerTotals[$groupKey])) {
                $customerTotals[$groupKey] = [
                    'name' => $displayName,
                    'total' => 0,
                    'lines' => []
                ];
            }

            $customerTotals[$groupKey]['total'] += $bill->TotalBill;
            $totalAmount += $bill->TotalBill;

            // Process each line item
            foreach ($bill->billLines as $line) {
                $lineId = $line->BLineID;
                $description = $line->BLineDesc;
                $amount = (float)$line->BLineAmount;
                
                // Track all unique line descriptions and their IDs
                if (!isset($lineDescriptions[$lineId])) {
                    $lineDescriptions[$lineId] = $description;
                }
                
                // Initialize the line if it doesn't exist
                if (!isset($customerTotals[$groupKey]['lines'][$lineId])) {
                    $customerTotals[$groupKey]['lines'][$lineId] = [
                        'amount' => 0,
                        'description' => $description
                    ];
                }
                
                // Add the amount to the line total
                $customerTotals[$groupKey]['lines'][$lineId]['amount'] += $amount;
            }
        }

        // Sort line descriptions by BLineID
        ksort($lineDescriptions);

        // Calculate column totals for all pages
        $columnTotals = [];
        foreach ($lineDescriptions as $lineId => $description) {
            $columnTotals[$lineId] = 0;
            
            foreach ($customerTotals as $customer) {
                if (isset($customer['lines'][$lineId])) {
                    $columnTotals[$lineId] += $customer['lines'][$lineId]['amount'];
                }
            }
        }

        $pdfData = [
            'company' => $company,
            'bulkInvoiceId' => $bulkInvoiceId,
            'billingPeriod' => $billingPeriod,
            'customerTotals' => $customerTotals,
            'lineDescriptions' => $lineDescriptions,
            'totalAmount' => $totalAmount,
            'columnTotals' => $columnTotals
        ];

        $pdf = PDF::loadView('reports.detail-report-2-pdf', $pdfData);
        $pdf->setPaper('A4', 'portrait');

        // Return as stream to open in new tab instead of download
        return $pdf->stream('detail-report-' . $bulkInvoiceId . '.pdf')
            ->withHeaders([
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="detail-report-' . $bulkInvoiceId . '.pdf"'
            ]);
    }
    
    public function exportBillSummary(Request $request): BinaryFileResponse
    {
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $propertyId = $request->input('property_id');
        
        return Excel::download(new BillSummaryExport($startDate, $endDate, $propertyId), 
            'bill-summary-' . now()->format('Y-m-d') . '.xlsx');
    }

    public function rentRollReport(Request $request)
{
    $date = $request->input('date', now()->format('Y-m-01'));
    $propertyId = $request->input('property_id');
    
    // Parse the report date
    $reportDate = \Carbon\Carbon::parse($date);
    $startOfMonth = $reportDate->copy()->startOfMonth();
    $endOfMonth = $reportDate->copy()->endOfMonth();
    
    $query = UnitAssignment::with([
        'unit.property',
        'customer',
        'payments' => function($q) use ($startOfMonth, $endOfMonth) {
            $q->whereBetween('PaymentDate', [$startOfMonth, $endOfMonth]);
        },
        'bills' => function($q) use ($startOfMonth, $endOfMonth) {
            $q->whereBetween('BillingDate', [$startOfMonth, $endOfMonth]);
        }
    ])
    ->where('start_date', '<=', $endOfMonth)
    ->where(function($q) use ($startOfMonth) {
        $q->whereNull('end_date')
          ->orWhere('end_date', '>=', $startOfMonth);
    })
    ->whereHas('unit', function($q) use ($propertyId) {
        if ($propertyId) {
            $q->where('PropID', $propertyId);
        }
    });
    
    $assignments = $query->get()->map(function($assignment) use ($startOfMonth, $endOfMonth) {
        // Debug: Log assignment and unit data
        Log::info('Processing assignment', [
            'assignment_id' => $assignment->id,
            'unit' => $assignment->unit ? $assignment->unit->toArray() : null
        ]);
        
        // Ensure the unit relationship is loaded
        if (!$assignment->relationLoaded('unit')) {
            $assignment->load('unit');
        }
        
        // Debug: Check if unit is loaded and has MonthlyRent
        if ($assignment->unit) {
            Log::info('Unit details', [
                'unit_id' => $assignment->unit->UnitID,
                'monthly_rent' => $assignment->unit->MonthlyRent,
                'unit_data' => $assignment->unit->toArray()
            ]);
        } else {
            Log::warning('No unit found for assignment', ['assignment_id' => $assignment->id]);
        }
        
        // Get payments for the specific month
        $payments = $assignment->payments()
            ->whereBetween('PaymentDate', [$startOfMonth, $endOfMonth])
            ->get();
            
        $assignment->paid_amount = $payments->sum('TotalPayment');
        $monthlyRent = $assignment->unit ? (float)$assignment->unit->MonthlyRent : 0;
        
        // Get total payable from rent service bills
        $totalPayable = $assignment->bills->sum('TotalPayable');
        $assignment->total_billed = $totalPayable > 0 ? $totalPayable : $monthlyRent;
        
        $assignment->outstanding_balance = max(0, $assignment->total_billed - $assignment->paid_amount);
        $assignment->last_payment_date = $payments->max('PaymentDate');
        
        // Debug: Log calculated values
        Log::info('Calculated values', [
            'paid_amount' => $assignment->paid_amount,
            'monthly_rent' => $monthlyRent,
            'outstanding_balance' => $assignment->outstanding_balance
        ]);
        
        return $assignment;
    });
    
    $properties = Property::all();
    
    // Calculate summary statistics with proper number formatting
    $totalRentDue = 0;
    $totalCollected = 0;
    $totalOutstanding = 0;
    $totalBilled = 0;
    
    foreach ($assignments as $assignment) {
        $monthlyRent = (float)($assignment->unit->MonthlyRent ?? 0);
        $totalBilled += $assignment->total_billed ?? $monthlyRent;
        $totalRentDue += $monthlyRent;
        $totalCollected += (float)$assignment->paid_amount;
        $totalOutstanding += (float)$assignment->outstanding_balance;
    }
    
    $collectionRate = $totalRentDue > 0 ? round(($totalCollected / $totalRentDue) * 100, 2) : 0;
    
    return view('reports.rent-roll', [
        'assignments' => $assignments,
        'properties' => $properties,
        'reportDate' => $reportDate,
        'selectedProperty' => $propertyId,
        'totalRentDue' => $totalRentDue,
        'totalBilled' => $totalBilled,
        'totalCollected' => $totalCollected,
        'totalOutstanding' => $totalOutstanding,
        'collectionRate' => $collectionRate
    ]);
}
    
 
    public function exportFinancialReport(Request $request): BinaryFileResponse
    {
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $propertyId = $request->input('property_id');
        $type = $request->input('type', 'xlsx');
        
        return Excel::download(
            new FinancialReportExport($startDate, $endDate, $propertyId),
            'financial-report-' . now()->format('Y-m-d') . '.' . $type,
            \Maatwebsite\Excel\Excel::XLSX
        );
    }
    
   
    public function exportRentRollReport(Request $request): BinaryFileResponse
    {
        $date = $request->input('date', now()->format('Y-m-01'));
        $propertyId = $request->input('property_id');
        $type = $request->input('type', 'xlsx');
        
        return Excel::download(
            new RentRollExport($date, $propertyId),
            'rent-roll-' . now()->format('Y-m-d') . '.' . $type,
            \Maatwebsite\Excel\Excel::XLSX
        );
    }
    
 

}
