<?php

namespace App\Http\Controllers;

use App\Models\SmsTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class SmsTemplateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $templates = SmsTemplate::latest()->paginate(10);
        return view('settings.sms-templates.index', compact('templates'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('settings.sms-templates.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
            'is_active' => 'boolean',
        ]);

        // Generate slug from name
        $validated['slug'] = Str::slug($validated['name']);
        $validated['variables'] = []; // Initialize empty variables array

        SmsTemplate::create($validated);

        return redirect()->route('sms-templates.index')
            ->with('success', 'SMS template created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(SmsTemplate $smsTemplate)
    {
        return view('settings.sms-templates.show', compact('smsTemplate'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(SmsTemplate $smsTemplate)
    {
        return view('settings.sms-templates.edit', compact('smsTemplate'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SmsTemplate $smsTemplate)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('sms_templates', 'name')->ignore($smsTemplate->id)],
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
            'is_active' => 'boolean',
        ]);

        // Update slug if name was changed
        if ($smsTemplate->name !== $validated['name']) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        $smsTemplate->update($validated);

        return redirect()->route('sms-templates.index')
            ->with('success', 'SMS template updated successfully.');
    }

    /**
     * Toggle the active status of the specified resource.
     */
    public function toggleStatus(SmsTemplate $smsTemplate)
    {
        $smsTemplate->update(['is_active' => !$smsTemplate->is_active]);
        
        $status = $smsTemplate->is_active ? 'enabled' : 'disabled';
        return back()->with('success', "SMS template has been {$status}.");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SmsTemplate $smsTemplate)
    {
        $smsTemplate->delete();
        
        return redirect()->route('sms-templates.index')
            ->with('success', 'SMS template deleted successfully.');
    }
}
