<?php

namespace App\Mail;

use App\Models\Customer;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class CustomerStatement extends Mailable
{
    use Queueable, SerializesModels;

    public $customer;
    public $statement;
    public $startDate;
    public $endDate;
    public $company;

    /**
     * Create a new message instance.
     *
     * @return void
     */
    public function __construct($customer, $statement, $startDate, $endDate, $company)
    {
        $this->customer = $customer;
        $this->statement = $statement;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->company = $company;
    }

    /**
     * Build the message.
     *
     * @return $this
     */
    public function build()
    {
        $pdf = $this->generatePdf();
        $filename = 'statement_' . $this->customer->AccountNo . '_' . $this->startDate->format('Y-m-d') . '_to_' . $this->endDate->format('Y-m-d') . '.pdf';
        
        return $this->subject('Your Account Statement - ' . $this->company['name'])
                    ->view('emails.customer-statement', [
                        'customer' => $this->customer,
                        'statement' => $this->statement,
                        'startDate' => $this->startDate,
                        'endDate' => $this->endDate,
                        'company' => $this->company
                    ])
                    ->attachData($pdf->output(), $filename, [
                        'mime' => 'application/pdf',
                    ]);
    }

    /**
     * Generate the PDF for the statement
     *
     * @return \Barryvdh\DomPDF\PDF
     */
    protected function generatePdf()
    {
        // Get company logo path
        $company = auth()->user()->company;
        $logoPath = null;
        
        if ($company->CompanyLogo) {
            // Check public/logo directory first
            $publicLogoPath = 'logo/' . $company->CompanyLogo;
            if (file_exists(public_path($publicLogoPath))) {
                $logoPath = public_path($publicLogoPath);
            } 
            // Check storage if not found in public/logo
            elseif (file_exists(storage_path('app/public/' . $company->CompanyLogo))) {
                $logoPath = storage_path('app/public/' . $company->CompanyLogo);
            }
        }
        
        // If no logo found, use default logo
        if (!$logoPath) {
            $logoPath = public_path('logo/sulispmslogo.png');
        }
        
        $companyData = array_merge($this->company, ['logo' => $logoPath]);
        
        return PDF::loadView('pdf.statement', [
            'customer' => $this->customer,
            'statement' => $this->statement,
            'startDate' => $this->startDate,
            'endDate' => $this->endDate,
            'company' => $companyData
        ]);
    }
}
