<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;


class Bill extends Model
{
    protected $table = 'bill';
    protected $primaryKey = 'BillID';
    public $incrementing = true;
    public $timestamps = false; // Disable automatic timestamp handling

    protected $fillable = [
        'BillingPeriod',
        'BulkInvoiceID',
        'CustomerID',
        'BillingDate',
        'MeterReadingID',
        'BillType',
        'CurReading',
        'PrvReading',
        'CurReadingDate',
        'PrvReadingDate',
        'DueDate',
        'ConsumptionBilled',
        'BalanceBF',
        'TotalPayments',
        'TotalBill',
        'BalanceCF',
        'LastUpdateDate',
        'LastUpdateUser',
        'BillQuality'
    ];

    protected $dates = [
        'BillingDate',
        'CurReadingDate',
        'PrvReadingDate',
        'DueDate',
        'LastUpdateDate'
    ];

    protected $appends = ['formatted_billing_period'];

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'CustomerID', 'CustomerID');
    }

    public function bulkInvoice()
    {
        return $this->belongsTo(BulkInvoice::class, 'BulkInvoiceID', 'BulkInvoiceID');
    }
    
    public function property()
    {
        return $this->hasOneThrough(
            Property::class,
            Customer::class,
            'CustomerID',  // Foreign key on customers table
            'PropID',      // Foreign key on properties table
            'CustomerID',  // Local key on bills table
            'UnitID'       // Local key on customers table
        )->join('prounits', 'prounits.UnitID', '=', 'customers.UnitID');
    }

    public function meterReading()
    {
        return $this->belongsTo(MeterReading::class, 'MeterReadingID', 'MeterReadingID');
    }

    public function billLines()
    {
        return $this->hasMany(BillLine::class, 'BillID', 'BillID');
    }

    /**
     * Get the formatted billing period.
     *
     * @return string
     */
    public function getFormattedBillingPeriodAttribute()
    {
        if (empty($this->BillingPeriod)) {
            return 'the current period';
        }

        try {
            // Convert the integer to a string and pad with leading zeros if necessary
            $period = str_pad($this->BillingPeriod, 6, '0', STR_PAD_LEFT);
            return \Carbon\Carbon::createFromFormat('Ym', $period)->format('F Y');
        } catch (\Exception $e) {
            // If we can't parse the date, return the raw value
            return (string) $this->BillingPeriod;
        }
    }

    /**
     * Set the billing period.
     *
     * @param  mixed  $value
     * @return void
     */
    public function setBillingPeriodAttribute($value)
    {
        if (empty($value)) {
            $this->attributes['BillingPeriod'] = null;
            return;
        }

        // If it's already an integer, store it as is
        if (is_numeric($value) && (int)$value == $value) {
            $this->attributes['BillingPeriod'] = (int)$value;
            return;
        }

        // Try to parse as a date
        try {
            $date = \Carbon\Carbon::parse($value);
            $this->attributes['BillingPeriod'] = (int)$date->format('Ym');
        } catch (\Exception $e) {
            // If we can't parse it, store as is
            $this->attributes['BillingPeriod'] = $value;
        }
    }
}