<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Scopes\CompanyScope;

class Customer extends Model
{
    
    protected $table = 'customer';
    protected $primaryKey = 'CustomerID';
    public $incrementing = true;
    public $timestamps = false;

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'ContractDate' => 'date:Y-m-d',
        'DepositDate' => 'date:Y-m-d',
        'WInitialReading' => 'decimal:2',
        'EInitialReading' => 'decimal:2',
        'WAvgCons' => 'decimal:2',
        'EAvgCons' => 'decimal:2',
        'AccountBalance' => 'decimal:2',
        'TotalDebit' => 'decimal:2',
        'TotalCredit' => 'decimal:2',
    ];

    /**
     * The "booting" method of the model.
     */
    protected static function booted()
    {
        static::addGlobalScope('company', function ($builder) {
            if (auth()->check() && !is_null(auth()->user()->CompanyID)) {
                $builder->where('CompanyID', auth()->user()->CompanyID);
            }
        });
    }
    
    /**
     * Get the company that owns the customer.
     */
    public function company()
    {
        return $this->belongsTo(Company::class, 'CompanyID', 'CompanyID');
    }

    /**
     * The property services that belong to the customer.
     */
    public function propServices()
    {
        return $this->belongsToMany(PropService::class, 'customer_prop_service', 'CustomerID', 'IDService')
            ->withPivot('Price', 'Quantity', 'StartDate', 'EndDate', 'IsActive', 'Notes')
            ->withTimestamps();
    }

    /**
     * Get all tenancies for the customer.
     */
    public function tenancies()
    {
        return $this->hasMany(Tenancy::class, 'customer_id', 'CustomerID')
            ->orderBy('start_date', 'desc');
    }

    /**
     * Get the current active tenancy.
     */
    public function currentTenancy()
    {
        return $this->hasOne(Tenancy::class, 'customer_id', 'CustomerID')
            ->where('status', 'active')
            ->latest('start_date');
    }

    /**
     * Get all units currently rented by the customer.
     */
    public function currentUnits()
    {
        return $this->hasManyThrough(
            ProUnit::class,
            Tenancy::class,
            'customer_id', // Foreign key on tenancies table
            'UnitID',      // Foreign key on prounits table
            'CustomerID',  // Local key on customers table
            'unit_id'      // Local key on tenancies table
        )->where('tenancies.status', 'active');
    }

    /**
     * Check if the customer is currently renting any unit.
     */
    public function isRenting(): bool
    {
        return $this->currentTenancy()->exists();
    }

    /**
     * Get the total rent amount for all active tenancies.
     */
    public function getTotalRentAttribute()
    {
        return $this->tenancies()
            ->active()
            ->sum('monthly_rent');
    }

    /**
     * Get the customer's full address.
     */
    public function getFullAddressAttribute()
    {
        $parts = array_filter([
            $this->Address1,
            $this->Address2,
            $this->Address3 ?? null,
        ]);

        return implode(', ', $parts);
    }

    /**
     * Scope a query to only include active customers.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('AccountStatus', 'A');
    }

    /**
     * Scope a query to only include customers with active tenancies.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeWithActiveTenancy($query)
    {
        return $query->whereHas('tenancies', function($q) {
            $q->where('status', 'active');
        });
    }

    protected $fillable = [
        'CustomerName', 'AccountNo', 'TelNo', 'Email','CustomerEmail', 'Address1', 'Address2', 'Address3',
        'ClientCode', 'TenantName', 'AccountType', 'ContractDate', 'AccountStatus',
        'Deposit', 'DepositDate', 'WMeterNo', 'WInitialReading', 'EMeterNo',
        'EInitialReading', 'WAvgCons', 'EAvgCons', 'AccountBalance', 'TotalDebit',
        'TotalCredit', 'UnitID', 'CompanyID', 'AccountFileName', 'TenantName',
        'CompanyID', 'AddressLine'
    ];

    /**
     * Get the unit associated with the customer.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function unit()
    {
        return $this->belongsTo(ProUnit::class, 'UnitID', 'UnitID');
    }

    public function unitAssignments()
    {
        return $this->hasMany(UnitAssignment::class, 'customer_id', 'CustomerID');
    }

    public function currentUnitAssignment()
    {
        return $this->hasOne(UnitAssignment::class, 'customer_id', 'CustomerID')
            ->where('status', 'active')
            ->latest('start_date');
    }

    public function bills()
    {
        return $this->hasMany(Bill::class, 'CustomerID', 'CustomerID');
    }

    public function meterReadings()
    {
        return $this->hasMany(MeterReading::class, 'CustomerID', 'CustomerID');
    }

    public function payments()
    {
        return $this->hasMany(Payment::class, 'CustomerID', 'CustomerID');
    }

    /**
     * Get the rent service bills for the customer.
     */
    public function rentServiceBills()
    {
        return $this->hasMany(\App\Models\RentServiceBill::class, 'CustomerID', 'CustomerID');
    }

    /**
     * Generate an account statement for the customer
     *
     * @param string|null $startDate Start date for the statement period (Y-m-d)
     * @param string|null $endDate End date for the statement period (Y-m-d)
     * @return array
     */
    public function generateAccountStatement($startDate = null, $endDate = null)
    {
        // Set default date range if not provided
        $endDate = $endDate ?: now()->format('Y-m-d');
        $startDate = $startDate ?: now()->subMonths(3)->format('Y-m-d');

        // Get all bills, rent service bills, and payments within the date range
        $bills = $this->bills()
            ->whereBetween('BillingDate', [$startDate, $endDate])
            ->orderBy('BillingDate', 'asc')
            ->get()
            ->map(function($bill) {
                $bill->bill_type = 'utility';
                return $bill;
            });

        $rentBills = $this->rentServiceBills()
            ->whereBetween('BillingDate', [$startDate, $endDate])
            ->orderBy('BillingDate', 'asc')
            ->get()
            ->map(function($bill) {
                $bill->bill_type = 'rent';
                $bill->BillType = 'Rent Service Bill'; // Ensure BillType is set
                return $bill;
            });

        // Merge all bills
        $allBills = $bills->merge($rentBills);

        $payments = $this->payments()
            ->whereBetween('PaymentDate', [$startDate, $endDate])
            ->orderBy('PaymentDate', 'asc')
            ->get();

        // Combine and sort all transactions by date
        $transactions = collect();

        // Add all bills as debit transactions
        foreach ($allBills as $bill) {
            $billType = $bill->bill_type ?? 'utility';
            $description = 'Bill #' . $bill->BillID . ' - ' . ($bill->BillType ?? 'Rent Service');
            
            // For rent service bills, use TotalAmount instead of TotalBill
            $amount = $bill->TotalAmount ?? $bill->TotalBill ?? 0;
            
            $transactions->push([
                'date' => $bill->BillingDate,
                'type' => 'bill',
                'bill_type' => $billType,
                'reference' => $bill->BillID,
                'description' => $description,
                'debit' => $amount,
                'credit' => 0,
                'balance' => 0, // Will be calculated later
                'entity' => $bill
            ]);
        }

        // Add payments as credit transactions
        foreach ($payments as $payment) {
            $transactions->push([
                'date' => $payment->PaymentDate,
                'type' => 'payment',
                'reference' => $payment->ReceiptNo,
                'description' => 'Payment - ' . $payment->PaymentMode,
                'debit' => 0,
                'credit' => $payment->TotalPayment,
                'balance' => 0, // Will be calculated later
                'entity' => $payment
            ]);
        }

        // Sort all transactions by date
        $transactions = $transactions->sortBy('date');

        // Calculate running balance
        $runningBalance = $this->AccountBalance ?? 0;
        
        // Apply opening balance as the first transaction
        $statement = [
            'customer' => [
                'id' => $this->CustomerID,
                'name' => $this->CustomerName,
                'account_number' => $this->AccountNo,
                'address' => $this->Address1 . ($this->Address2 ? ', ' . $this->Address2 : ''),
                'phone' => $this->TelNo,
                'email' => $this->Email,
                'opening_balance' => $runningBalance
            ],
            'period' => [
                'from' => $startDate,
                'to' => $endDate
            ],
            'transactions' => [],
            'summary' => [
                'opening_balance' => $runningBalance,
                'total_debits' => 0,
                'total_credits' => 0,
                'closing_balance' => 0
            ]
        ];

        // Add opening balance as first transaction
        $statement['transactions'][] = [
            'date' => $startDate,
            'type' => 'opening_balance',
            'reference' => 'OPENING',
            'description' => 'Opening Balance',
            'debit' => 0,
            'credit' => 0,
            'balance' => $runningBalance,
            'is_summary' => true
        ];

        // Process each transaction and calculate running balance
        foreach ($transactions as $transaction) {
            $runningBalance += $transaction['debit'] - $transaction['credit'];
            $transaction['balance'] = $runningBalance;
            $statement['transactions'][] = $transaction;
            
            // Update summary
            $statement['summary']['total_debits'] += $transaction['debit'];
            $statement['summary']['total_credits'] += $transaction['credit'];
        }

        // Update closing balance in summary
        $statement['summary']['closing_balance'] = $runningBalance;

        return $statement;
    }
}