<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class ProUnit extends Model
{
    public $timestamps = false;
    
    protected $table = 'prounits';
    protected $primaryKey = 'UnitID';
    public $incrementing = true;

    protected $fillable = [
        'PropID',
        'BlockID',
        'UnitIdentity',
        'BlockNo',
        'FloorNo',
        'DoorNo',
        'UnitDesc',
        'EFixedCharge',
        'WFixedCharge',
        'MonthlyRent',
        'WMeterNo',
        'EMeterNo',
        'WLastReading',
        'ELastReading',
        'UnitActive',
        'WMaxReading',
        'EMaxReading',
        'OccupationStatus'
    ];

    protected $casts = [
        'UnitActive' => 'boolean',
        'MonthlyRent' => 'float',
        'EFixedCharge' => 'float',
        'WFixedCharge' => 'float',
        'WLastReading' => 'float',
        'ELastReading' => 'float',
        'WMaxReading' => 'integer',
        'EMaxReading' => 'integer'
    ];
    
    protected $attributes = [
        'MonthlyRent' => 0.00,
        'EFixedCharge' => 0.00,
        'WFixedCharge' => 0.00,
        'WLastReading' => 0.00,
        'ELastReading' => 0.00,
        'WMaxReading' => 0,
        'EMaxReading' => 0
    ];

    /**
     * Get the property that owns the unit.
     */
    public function property()
    {
        return $this->belongsTo(Property::class, 'PropID', 'PropID');
    }

    /**
     * Get the block that owns the unit.
     */
    public function block()
    {
        return $this->belongsTo(Block::class, 'BlockID', 'BlockID');
    }

    public function customers()
    {
        return $this->hasMany(Customer::class, 'UnitID', 'UnitID');
    }

    public function unitAssignments()
    {
        return $this->hasMany(UnitAssignment::class, 'unit_id', 'UnitID');
    }

    public function currentAssignments()
    {
        return $this->hasMany(UnitAssignment::class, 'unit_id', 'UnitID')
            ->where('status', 'active')
            ->with('customer')
            ->orderBy('start_date', 'desc');
    }

    /**
     * Get the active assignments for the unit.
     */
    public function activeAssignments()
    {
        return $this->hasMany(UnitAssignment::class, 'unit_id', 'UnitID')
            ->where('status', 'active');
    }

    /**
     * Get the active customer for the unit.
     */
    /**
     * Get the current active tenancy for the unit.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function currentTenancy()
    {
        return $this->hasOne(Tenancy::class, 'unit_id')
            ->where('status', 'active')
            ->latest('start_date');
    }

    /**
     * Get all tenancies for the unit.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function tenancies()
    {
        return $this->hasMany(Tenancy::class, 'unit_id')
            ->orderBy('start_date', 'desc');
    }

    /**
     * Get the current tenant (customer) for the unit.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOneThrough
     */
    public function currentTenant()
    {
        return $this->hasOneThrough(
            Customer::class,
            Tenancy::class,
            'unit_id',      // Foreign key on tenancies table
            'CustomerID',   // Foreign key on customers table
            'UnitID',       // Local key on prounits table
            'customer_id'   // Local key on tenancies table
        )->where('tenancies.status', 'active');
    }

    /**
     * Check if the unit is currently occupied.
     *
     * @return bool
     */
    public function getIsOccupiedAttribute()
    {
        return $this->currentTenancy()->exists();
    }

    /**
     * Get the active customer for the unit.
     * 
     * @deprecated Use currentTenant() instead
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function activeCustomer()
    {
        return $this->hasOne(Customer::class, 'UnitID', 'UnitID')
            ->where('customer.AccountStatus', 'Active')
            ->select('customer.CustomerID', 'customer.CustomerName', 'customer.TelNo');
    }
}