<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;

class Property extends Model
{
    public $timestamps = false;
    
    protected $table = 'property';
    protected $primaryKey = 'PropID';
    public $incrementing = true;
    
    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'PropID';
    }

    protected $fillable = [
        'PropName',
        'PropDesc',
        'BillPeriodicity',
        'City',
        'PropertyType',
        'NoMonthPerYear',
        'WMeterNo',
        'EMeterNo',
        'CompanyID'
    ];

    /**
     * The "booting" method of the model.
     */
    protected static function booted()
    {
        static::addGlobalScope('company', function ($builder) {
            if (auth()->check() && !is_null(auth()->user()->CompanyID)) {
                $builder->where('CompanyID', auth()->user()->CompanyID);
            }
        });
    }

    /**
     * Get the company that owns the property.
     */
    public function company()
    {
        return $this->belongsTo(Company::class, 'CompanyID', 'CompanyID');
    }

    public function units()
    {
        return $this->hasMany(ProUnit::class, 'PropID', 'PropID')
            ->when(auth()->check() && !is_null(auth()->user()->CompanyID), function($query) {
                $query->where('prounits.CompanyID', auth()->user()->CompanyID);
            });
    }

    /**
     * Get all blocks for the property.
     */
    public function blocks()
    {
        return $this->hasMany(Block::class, 'PropID', 'PropID');
    }

    /**
     * Get the active customers for the property.
     */
    public function activeCustomers()
    {
        return $this->hasManyThrough(
            Customer::class,
            ProUnit::class,
            'PropID', // Foreign key on the prounits table
            'UnitID',  // Foreign key on the customers table
            'PropID',  // Local key on the properties table
            'UnitID'   // Local key on the prounits table
        )->where('customer.AccountStatus', 'Active')
         ->when(auth()->check() && !is_null(auth()->user()->CompanyID), function($query) {
             $query->where('customer.CompanyID', auth()->user()->CompanyID);
         });
    }

    /**
     * Get all services for the property.
     */
    public function services()
    {
        return $this->hasMany(PropService::class, 'PropID', 'PropID')
            ->when(auth()->check() && !is_null(auth()->user()->CompanyID), function($query) {
                $query->where('propservice.CompanyID', auth()->user()->CompanyID);
            });
    }
}