<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up()
    {
        // Unit Types Table
        Schema::create('unit_types', function (Blueprint $table) {
            $table->increments('UnitTypeID');
            $table->string('Name', 50);
            $table->string('Description')->nullable();
            $table->timestamps();
        });

        // Service Types Table
        Schema::create('service_types', function (Blueprint $table) {
            $table->increments('ServiceTypeID');
            $table->string('Name', 50);
            $table->enum('BillingType', ['recurring', 'one_time'])->default('recurring');
            $table->boolean('IsUtility')->default(false);
            $table->boolean('IsTaxable')->default(true);
            $table->timestamps();
        });

        // Billing Frequencies Table
        Schema::create('billing_frequencies', function (Blueprint $table) {
            $table->increments('BillingFrequencyID');
            $table->string('Name', 50);
            $table->integer('Months')->default(1);
            $table->boolean('IsActive')->default(true);
            $table->timestamps();
        });

        // Payment Methods Table
        Schema::create('payment_methods', function (Blueprint $table) {
            $table->increments('PaymentMethodID');
            $table->string('Name', 50);
            $table->string('AccountNumber')->nullable();
            $table->string('AccountName')->nullable();
            $table->string('BankName')->nullable();
            $table->string('Branch')->nullable();
            $table->boolean('IsActive')->default(true);
            $table->timestamps();
        });

        // Insert default values
        $this->seedLookupTables();
    }

    public function down()
    {
        Schema::dropIfExists('payment_methods');
        Schema::dropIfExists('billing_frequencies');
        Schema::dropIfExists('service_types');
        Schema::dropIfExists('unit_types');
    }

    /**
     * Seed the lookup tables with default values
     */
    private function seedLookupTables()
    {
        // Unit Types
        $unitTypes = [
            ['Name' => 'Bedsitter'],
            ['Name' => '1 Bedroom'],
            ['Name' => '2 Bedroom'],
            ['Name' => '3 Bedroom'],
            ['Name' => '4 Bedroom'],
            ['Name' => 'Office'],
            ['Name' => 'Shop'],
            ['Name' => 'Warehouse'],
            ['Name' => 'Studio'],
            ['Name' => 'Penthouse']
        ];

        foreach ($unitTypes as $type) {
            DB::table('unit_types')->insert($type);
        }

        // Service Types
        $serviceTypes = [
            ['Name' => 'Rent', 'BillingType' => 'recurring', 'IsUtility' => false, 'IsTaxable' => true],
            ['Name' => 'Water', 'BillingType' => 'recurring', 'IsUtility' => true, 'IsTaxable' => true],
            ['Name' => 'Electricity', 'BillingType' => 'recurring', 'IsUtility' => true, 'IsTaxable' => true],
            ['Name' => 'Security', 'BillingType' => 'recurring', 'IsUtility' => false, 'IsTaxable' => true],
            ['Name' => 'Garbage', 'BillingType' => 'recurring', 'IsUtility' => false, 'IsTaxable' => true],
            ['Name' => 'Maintenance', 'BillingType' => 'recurring', 'IsUtility' => false, 'IsTaxable' => true],
            ['Name' => 'Deposit', 'BillingType' => 'one_time', 'IsUtility' => false, 'IsTaxable' => false],
            ['Name' => 'Registration', 'BillingType' => 'one_time', 'IsUtility' => false, 'IsTaxable' => true],
            ['Name' => 'Connection Fee', 'BillingType' => 'one_time', 'IsUtility' => true, 'IsTaxable' => true]
        ];

        foreach ($serviceTypes as $type) {
            DB::table('service_types')->insert($type);
        }

        // Billing Frequencies
        $frequencies = [
            ['Name' => 'Monthly', 'Months' => 1],
            ['Name' => 'Quarterly', 'Months' => 3],
            ['Name' => 'Semi-Annually', 'Months' => 6],
            ['Name' => 'Annually', 'Months' => 12],
            ['Name' => 'Bi-Monthly', 'Months' => 2]
        ];

        foreach ($frequencies as $frequency) {
            DB::table('billing_frequencies')->insert($frequency);
        }

        // Payment Methods
        $paymentMethods = [
            ['Name' => 'Cash', 'IsActive' => true],
            ['Name' => 'M-Pesa', 'IsActive' => true],
            ['Name' => 'Bank Transfer', 'IsActive' => true],
            ['Name' => 'Check', 'IsActive' => true],
            ['Name' => 'Credit Card', 'IsActive' => true],
            ['Name' => 'Debit Card', 'IsActive' => true],
            ['Name' => 'Bank Deposit', 'IsActive' => true]
        ];

        foreach ($paymentMethods as $method) {
            DB::table('payment_methods')->insert($method);
        }
    }
};
