@extends('customers.layout')

@section('customer-content')
    <div class="bg-gradient-to-r from-blue-700 to-blue-800 shadow-lg rounded-lg mb-8 overflow-hidden">
        <div class="px-6 py-4">
            <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center">
                <div>
                    <h2 class="text-2xl font-bold text-yellow-400">
                        <i class="fas fa-user-edit mr-2"></i>Edit Customer
                    </h2>
                    <p class="mt-1 text-sm text-blue-200">
                        Update the details for {{ $customer->CustomerName }}
                    </p>
                </div>
                <div class="mt-4 sm:mt-0">
                    <a href="{{ route('customers.show', $customer->CustomerID) }}" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-blue-800 bg-yellow-400 hover:bg-yellow-300 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 transition duration-150 ease-in-out">
                        <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                        </svg>
                        Back to Customer
                    </a>
                </div>
            </div>
        </div>
    </div>
        
    <div class="bg-white shadow-xl rounded-lg overflow-hidden">
        <form action="{{ route('customers.update', $customer->CustomerID) }}" method="POST" class="p-6">
            @csrf
            @method('PUT')
            
            <div class="space-y-6">
                <!-- Personal Information -->
                <div>
                    <div class="flex items-center mb-4 pb-2 border-b border-gray-200">
                        <div class="flex-shrink-0 bg-blue-100 p-1.5 rounded-full mr-3">
                            <svg class="h-5 w-5 text-blue-700" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                            </svg>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800">
                            Personal Information
                        </h4>
                    </div>
                    
                    <div class="grid grid-cols-1 gap-y-6 gap-x-4 sm:grid-cols-6">
                        <div class="sm:col-span-3">
                            <label for="CustomerName" class="block text-sm font-medium text-gray-700">
                                Full Name <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <input type="text" name="CustomerName" id="CustomerName" autocomplete="name" 
                                    value="{{ old('CustomerName', $customer->CustomerName) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('CustomerName') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('CustomerName')
                                    <p class="mt-2 text-sm text-red-600" id="name-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="TenantName" class="block text-sm font-medium text-gray-700">
                                Tenant Name (if different)
                            </label>
                            <div class="mt-1">
                                <input type="text" name="TenantName" id="TenantName" 
                                    value="{{ old('TenantName', $customer->TenantName) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md">
                                @error('TenantName')
                                    <p class="mt-2 text-sm text-red-600" id="tenant-name-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="TelNo" class="block text-sm font-medium text-gray-700">
                                Phone Number <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <input type="text" name="TelNo" id="TelNo" autocomplete="tel" 
                                    value="{{ old('TelNo', $customer->TelNo) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('TelNo') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('TelNo')
                                    <p class="mt-2 text-sm text-red-600" id="tel-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="Email" class="block text-sm font-medium text-gray-700">
                                Email Address
                            </label>
                            <div class="mt-1">
                                <input type="email" name="Email" id="Email" autocomplete="email" 
                                    value="{{ old('Email', $customer->Email) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('Email') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('Email')
                                    <p class="mt-2 text-sm text-red-600" id="email-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="Address1" class="block text-sm font-medium text-gray-700">
                                Address Line 1 <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <input type="text" name="Address1" id="Address1" autocomplete="address-line1" 
                                    value="{{ old('Address1', $customer->Address1) }}"
                                    class="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md @error('Address1') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('Address1')
                                    <p class="mt-2 text-sm text-red-600" id="address1-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="Address2" class="block text-sm font-medium text-gray-700">
                                Address Line 2
                            </label>
                            <div class="mt-1">
                                <input type="text" name="Address2" id="Address2" autocomplete="address-line2" 
                                    value="{{ old('Address2', $customer->Address2) }}"
                                    class="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md">
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="Address3" class="block text-sm font-medium text-gray-700">
                                City/Town
                            </label>
                            <div class="mt-1">
                                <input type="text" name="Address3" id="Address3" autocomplete="address-level2" 
                                    value="{{ old('Address3', $customer->Address3) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md">
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="ClientCode" class="block text-sm font-medium text-gray-700">
                                Client/ID Number
                            </label>
                            <div class="mt-1">
                                <input type="text" name="ClientCode" id="ClientCode" 
                                    value="{{ old('ClientCode', $customer->ClientCode) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md">
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Account Information -->
                <div class="mt-8">
                    <div class="flex items-center mb-4 pb-2 border-b border-gray-200">
                        <div class="flex-shrink-0 bg-blue-100 p-1.5 rounded-full mr-3">
                            <svg class="h-5 w-5 text-blue-700" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
                            </svg>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800">
                            Account Information
                        </h4>
                    </div>
                    
                    <div class="grid grid-cols-1 gap-y-6 gap-x-4 sm:grid-cols-6">
                        <div class="sm:col-span-3">
                            <label for="AccountNo" class="block text-sm font-medium text-gray-700">
                                Account Number <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <input type="text" name="AccountNo" id="AccountNo" 
                                    value="{{ old('AccountNo', $customer->AccountNo) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('AccountNo') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('AccountNo')
                                    <p class="mt-2 text-sm text-red-600" id="account-no-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="AccountType" class="block text-sm font-medium text-gray-700">
                                Account Type <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <select id="AccountType" name="AccountType" class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('AccountType') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                    <option value="RESIDENTIAL" {{ old('AccountType', $customer->AccountType) == 'RESIDENTIAL' ? 'selected' : '' }}>Residential</option>
                                    <option value="COMMERCIAL" {{ old('AccountType', $customer->AccountType) == 'COMMERCIAL' ? 'selected' : '' }}>Commercial</option>
                                </select>
                                @error('AccountType')
                                    <p class="mt-2 text-sm text-red-600" id="account-type-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="ContractDate" class="block text-sm font-medium text-gray-700">
                                Contract Date <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <input type="date" name="ContractDate" id="ContractDate" 
                                    value="{{ old('ContractDate', $customer->ContractDate ? $customer->ContractDate->format('Y-m-d') : now()->format('Y-m-d')) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('ContractDate') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('ContractDate')
                                    <p class="mt-2 text-sm text-red-600" id="contract-date-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="AccountStatus" class="block text-sm font-medium text-gray-700">
                                Account Status <span class="text-red-500">*</span>
                            </label>
                            <div class="mt-1">
                                <select id="AccountStatus" name="AccountStatus" class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('AccountStatus') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                    <option value="A" {{ old('AccountStatus', $customer->AccountStatus) == 'A' ? 'selected' : '' }}>Active</option>
                                    <option value="I" {{ old('AccountStatus', $customer->AccountStatus) == 'I' ? 'selected' : '' }}>Inactive</option>
                                </select>
                                @error('AccountStatus')
                                    <p class="mt-2 text-sm text-red-600" id="account-status-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Deposit Information -->
                <div class="mt-8">
                    <div class="flex items-center mb-4 pb-2 border-b border-gray-200">
                        <div class="flex-shrink-0 bg-blue-100 p-1.5 rounded-full mr-3">
                            <svg class="h-5 w-5 text-blue-700" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z" />
                            </svg>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800">
                            Deposit Information
                        </h4>
                    </div>
                    
                    <div class="grid grid-cols-1 gap-y-6 gap-x-4 sm:grid-cols-6">
                        <div class="sm:col-span-3">
                            <label for="Deposit" class="block text-sm font-medium text-gray-700">
                                Deposit Amount
                            </label>
                            <div class="mt-1 relative rounded-md shadow-sm">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500 sm:text-sm">Ksh</span>
                                </div>
                                <input type="number" name="Deposit" id="Deposit" step="0.01" min="0" 
                                    value="{{ old('Deposit', $customer->Deposit) }}"
                                    class="focus:ring-indigo-500 focus:border-indigo-500 block w-full pl-12 sm:text-sm border-gray-300 rounded-md @error('Deposit') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('Deposit')
                                    <p class="mt-2 text-sm text-red-600" id="deposit-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="DepositDate" class="block text-sm font-medium text-gray-700">
                                Deposit Date
                            </label>
                            <div class="mt-1">
                                <input type="date" name="DepositDate" id="DepositDate" 
                                    value="{{ old('DepositDate', $customer->DepositDate ? $customer->DepositDate->format('Y-m-d') : '') }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md @error('DepositDate') border-red-300 text-red-900 placeholder-red-300 focus:outline-none focus:ring-red-500 focus:border-red-500 @enderror">
                                @error('DepositDate')
                                    <p class="mt-2 text-sm text-red-600" id="deposit-date-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Unit Assignment -->
                <div class="mt-8">
                    <div class="flex items-center mb-4 pb-2 border-b border-gray-200">
                        <div class="flex-shrink-0 bg-blue-100 p-1.5 rounded-full mr-3">
                            <svg class="h-5 w-5 text-blue-700" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                            </svg>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800">
                            Unit Assignment
                        </h4>
                    </div>
                    
                    <div class="grid grid-cols-1 gap-y-6 gap-x-4 sm:grid-cols-6">
                        <div class="sm:col-span-6">
                            <label for="UnitID" class="block text-sm font-medium text-gray-700">
                                Assign to Unit (Optional)
                            </label>
                            <div class="mt-1">
                                <select id="UnitID" name="UnitID" class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md">
                                    <option value="">-- Select a Unit --</option>
                                    @foreach($units as $id => $unitData)
                                        <option value="{{ $id }}" 
                                                {{ old('UnitID', $customer->UnitID) == $id ? 'selected' : '' }}
                                                data-property-id="{{ $unitData['data-property-id'] ?? '' }}"
                                                data-water-meter="{{ $unitData['data-water-meter'] ?? '' }}"
                                                data-electricity-meter="{{ $unitData['data-electricity-meter'] ?? '' }}">
                                            {{ is_array($unitData) ? $unitData['text'] : $unitData }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <p class="mt-2 text-sm text-gray-500">
                                You can assign this customer to a unit now or do it later from the unit management section.
                            </p>
                            @if($customer->unit)
                                <div class="mt-2 p-3 bg-blue-50 rounded-md">
                                    <p class="text-sm text-blue-700">
                                        Currently assigned to: 
                                        <a href="{{ route('properties.units.edit', [$customer->unit->property->PropID, $customer->unit->UnitID]) }}" class="font-medium text-blue-800 hover:text-blue-900">
                                            {{ $customer->unit->UnitIdentity }} ({{ $customer->unit->property->PropName }})
                                        </a>
                                    </p>
                                </div>
                            @endif
                        </div>
                    </div>
                </div>
                
                <!-- Meter Information -->
                <div id="meterInformationSection" class="mt-8 {{ !$customer->UnitID ? 'hidden' : '' }}">
                    <div class="flex items-center mb-4 pb-2 border-b border-gray-200">
                        <div class="flex-shrink-0 bg-blue-100 p-1.5 rounded-full mr-3">
                            <svg class="h-5 w-5 text-blue-700" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                            </svg>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800">
                            Meter Information
                        </h4>
                    </div>
                    
                    <div class="grid grid-cols-1 gap-y-6 gap-x-4 sm:grid-cols-6">
                        <div class="sm:col-span-3">
                            <label for="WMeterNo" class="block text-sm font-medium text-gray-700">
                                Water Meter Number
                            </label>
                            <div class="mt-1">
                                <input type="text" name="WMeterNo" id="WMeterNo" 
                                    value="{{ old('WMeterNo', $customer->WMeterNo) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md bg-gray-100"
                                    {{ $customer->UnitID ? 'readonly' : '' }}>
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="WInitialReading" class="block text-sm font-medium text-gray-700">
                                Water Initial Reading
                            </label>
                            <div class="mt-1">
                                <input type="number" name="WInitialReading" id="WInitialReading" min="0" step="0.001"
                                    value="{{ old('WInitialReading', $customer->WInitialReading) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md">
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="EMeterNo" class="block text-sm font-medium text-gray-700">
                                Electricity Meter Number
                            </label>
                            <div class="mt-1">
                                <input type="text" name="EMeterNo" id="EMeterNo" 
                                    value="{{ old('EMeterNo', $customer->EMeterNo) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md bg-gray-100"
                                    {{ $customer->UnitID ? 'readonly' : '' }}>
                            </div>
                        </div>
                        
                        <div class="sm:col-span-3">
                            <label for="EInitialReading" class="block text-sm font-medium text-gray-700">
                                Electricity Initial Reading
                            </label>
                            <div class="mt-1">
                                <input type="number" name="EInitialReading" id="EInitialReading" min="0" step="0.01"
                                    value="{{ old('EInitialReading', $customer->EInitialReading) }}"
                                    class="shadow-sm focus:ring-indigo-500 focus:border-indigo-500 block w-full sm:text-sm border-gray-300 rounded-md">
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Services Assignment -->
                <div id="servicesSection" class="mt-8 {{ !$customer->UnitID ? 'hidden' : '' }}">
                    <div class="flex items-center mb-4 pb-2 border-b border-gray-200">
                        <div class="flex-shrink-0 bg-blue-100 p-1.5 rounded-full mr-3">
                            <svg class="h-5 w-5 text-blue-700" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                            </svg>
                        </div>
                        <h4 class="text-lg font-semibold text-gray-800">
                            Services Assignment
                        </h4>
                    </div>
                    
                    <!-- Services will be loaded here dynamically -->
                    <div id="servicesContainer" class="space-y-4">
                        @if($customer->UnitID && $customer->unit && $customer->unit->property)
                            @php
                                $propertyId = $customer->unit->property->PropID;
                                $services = \App\Models\PropService::where('PropID', $propertyId)
                                    ->where('CompanyID', $customer->CompanyID)
                                    ->orderBy('ServiceName')
                                    ->get();
                                 
                                $customerServices = $customer->propServices->keyBy('IDService');
                            @endphp
                             
                            @forelse($services as $service)
                                @php
                                    $isChecked = $customerServices->has($service->IDService);
                                    $pivot = $customerServices->get($service->IDService);
                                    $price = $isChecked ? $pivot->pivot->Price : $service->Price;
                                    $quantity = $isChecked ? $pivot->pivot->Quantity : 1;
                                    $startDate = $isChecked ? $pivot->pivot->StartDate : now()->format('Y-m-d');
                                @endphp
                                <div class="border border-gray-200 rounded-lg p-4 mt-2 service-item" id="service-{{ $service->IDService }}">
                                    <div class="flex items-start">
                                        <div class="flex items-center h-5">
                                            <input type="checkbox" 
                                                   name="services[{{ $service->IDService }}][id]" 
                                                   value="{{ $service->IDService }}" 
                                                   id="service_{{ $service->IDService }}" 
                                                   class="service-checkbox focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 rounded"
                                                   data-price="{{ $service->Price }}"
                                                   data-billing-cycle="MONTHLY"
                                                   {{ $isChecked ? 'checked' : '' }}>
                                        </div>
                                        <div class="ml-3 text-sm">
                                            <label for="service_{{ $service->IDService }}" class="font-medium text-gray-700">
                                                {{ $service->ServiceName }}
                                            </label>
                                            <div class="mt-3 grid grid-cols-1 md:grid-cols-4 gap-4">
                                                <div>
                                                    <label class="block text-xs font-medium text-gray-500">Unit Price</label>
                                                    <div class="mt-1">
                                                        <input type="number" 
                                                               name="services[{{ $service->IDService }}][price]" 
                                                               value="{{ $price }}" 
                                                               min="0" 
                                                               step="0.01"
                                                               class="service-price shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                                                               {{ $isChecked ? '' : 'disabled' }}
                                                               readonly>
                                                    </div>
                                                </div>
                                                <div>
                                                    <label class="block text-xs font-medium text-gray-500">Quantity</label>
                                                    <div class="mt-1">
                                                        <input type="number" 
                                                               name="services[{{ $service->IDService }}][quantity]" 
                                                               value="{{ $quantity }}" 
                                                               min="1"
                                                               class="service-quantity shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                                                               {{ $isChecked ? '' : 'disabled' }}>
                                                    </div>
                                                </div>
                                                <div>
                                                    <label class="block text-xs font-medium text-gray-500">Total</label>
                                                    <div class="mt-1">
                                                        <input type="text" 
                                                               name="services[{{ $service->IDService }}][display_total]"
                                                               class="service-total shadow-sm bg-gray-100 block w-full sm:text-sm border-gray-300 rounded-md"
                                                               value="{{ number_format($price * $quantity, 2) }}" 
                                                               readonly>
                                                    </div>
                                                </div>
                                                <div>
                                                    <label class="block text-xs font-medium text-gray-500">Start Date</label>
                                                    <div class="mt-1">
                                                        <input type="date" 
                                                               name="services[{{ $service->IDService }}][start_date]" 
                                                               class="service-start-date shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                                                               value="{{ $startDate }}"
                                                               {{ $isChecked ? '' : 'disabled' }}>
                                                    </div>
                                                </div>
                                                <input type="hidden" name="services[{{ $service->IDService }}][billing_cycle]" value="MONTHLY" class="service-billing-cycle">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @empty
                                <div class="bg-gray-50 p-4 rounded-lg text-center text-gray-500">
                                    No services available for this property.
                                </div>
                            @endforelse
                        @else
                            <div class="bg-gray-50 p-4 rounded-lg text-center text-gray-500">
                                Please select a unit to view available services.
                            </div>
                        @endif
                    </div>
                    
                    <!-- Loading indicator -->
                    <div id="servicesLoading" class="hidden flex justify-center items-center py-4">
                        <svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-blue-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        <span>Loading services...</span>
                    </div>
                </div>
                
                <!-- Form Actions -->
                <div class="pt-5">
                    <div class="flex justify-between">
                        <!-- <div>
                            <button type="button" 
                                onclick="if(confirm('Are you sure you want to delete this customer? This action cannot be undone.')) { document.getElementById('delete-form').submit(); }" 
                                class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500">
                                <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fill-rule="evenodd" d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z" clip-rule="evenodd" />
                                </svg>
                                Delete Customer
                            </button>
                        </div> -->
                        <div class="flex">
                            <a href="{{ route('customers.show', $customer->CustomerID) }}" class="bg-white py-2 px-4 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                                Cancel
                            </a>
                            <button type="submit" class="ml-3 inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                                Update Customer
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
    
    <!-- Delete Form (Hidden) -->
    <!-- <form id="delete-form" action="{{ route('customers.destroy', $customer->CustomerID) }}" method="POST" class="hidden">
        @csrf
        @method('DELETE')
    </form> -->
    
    @push('styles')
    <style>
        /* Custom styles for better form elements */
        select, input[type="text"], input[type="email"], input[type="tel"], input[type="date"], textarea {
            transition: all 0.2s ease-in-out;
        }
        select:focus, input:focus, textarea:focus {
            border-color: #1d4ed8;
            box-shadow: 0 0 0 3px rgba(29, 78, 216, 0.1);
        }
        /* Style for required field indicators */
        .required:after {
            content: ' *';
            color: #dc2626;
        }
    </style>
    @endpush
    
    @push('scripts')
    <script>
        // Function to load services for a property
        function loadServices(propertyId) {
            const servicesContainer = document.getElementById('servicesContainer');
            const servicesLoading = document.getElementById('servicesLoading');
            const servicesSection = document.getElementById('servicesSection');
            
            if (!propertyId) {
                servicesSection.classList.add('hidden');
                return;
            }
            
            servicesSection.classList.remove('hidden');
            servicesContainer.innerHTML = '';
            servicesLoading.classList.remove('hidden');
            
            fetch(`/properties/${propertyId}/services`, {
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                servicesLoading.classList.add('hidden');
                
                if (data.success && data.data && data.data.length > 0) {
                    const servicesHtml = data.data.map(service => {
                        const isChecked = {{ $customer->propServices->contains('IDService', '${service.IDService}') ? 'true' : 'false' }};
                        const pivot = {!! $customer->propServices->firstWhere('IDService', '${service.IDService}') ?: 'null' !!};
                        const price = pivot ? pivot.pivot.Price : service.Price;
                        const quantity = pivot ? pivot.pivot.Quantity : 1;
                        const startDate = pivot ? pivot.pivot.StartDate : new Date().toISOString().split('T')[0];
                        
                        return `
                            <div class="border border-gray-200 rounded-lg p-4 mt-2 service-item" id="service-${service.IDService}">
                                <div class="flex items-start">
                                    <div class="flex items-center h-5">
                                        <input type="checkbox" 
                                               name="services[${service.IDService}][id]" 
                                               value="${service.IDService}" 
                                               id="service_${service.IDService}" 
                                               class="service-checkbox focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 rounded"
                                               data-price="${service.Price}"
                                               data-billing-cycle="MONTHLY"
                                               ${isChecked ? 'checked' : ''}>
                                    </div>
                                    <div class="ml-3 text-sm">
                                        <label for="service_${service.IDService}" class="font-medium text-gray-700">
                                            ${service.ServiceName}
                                        </label>
                                        <div class="mt-3 grid grid-cols-1 md:grid-cols-4 gap-4">
                                            <div>
                                                <label class="block text-xs font-medium text-gray-500">Unit Price</label>
                                                <div class="mt-1">
                                                    <input type="number" 
                                                           name="services[${service.IDService}][price]" 
                                                           value="${price}" 
                                                           min="0" 
                                                           step="0.01"
                                                           class="service-price shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                                                           ${isChecked ? '' : 'disabled'}
                                                           readonly>
                                                </div>
                                            </div>
                                            <div>
                                                <label class="block text-xs font-medium text-gray-500">Quantity</label>
                                                <div class="mt-1">
                                                    <input type="number" 
                                                           name="services[${service.IDService}][quantity]" 
                                                           value="${quantity}" 
                                                           min="1"
                                                           class="service-quantity shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                                                           ${isChecked ? '' : 'disabled'}>
                                                </div>
                                            </div>
                                            <div>
                                                <label class="block text-xs font-medium text-gray-500">Total</label>
                                                <div class="mt-1">
                                                    <input type="text" 
                                                           name="services[${service.IDService}][display_total]"
                                                           class="service-total shadow-sm bg-gray-100 block w-full sm:text-sm border-gray-300 rounded-md"
                                                           value="${(price * quantity).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}" 
                                                           readonly>
                                                </div>
                                            </div>
                                            <div>
                                                <label class="block text-xs font-medium text-gray-500">Start Date</label>
                                                <div class="mt-1">
                                                    <input type="date" 
                                                           name="services[${service.IDService}][start_date]" 
                                                           class="service-start-date shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                                                           value="${startDate}"
                                                           ${isChecked ? '' : 'disabled'}>
                                                </div>
                                            </div>
                                            <input type="hidden" name="services[${service.IDService}][billing_cycle]" value="MONTHLY" class="service-billing-cycle">
                                        </div>
                                    </div>
                                </div>
                            `;
                    }).join('');
                    
                    servicesContainer.innerHTML = servicesHtml;
                } else {
                    servicesContainer.innerHTML = `
                        <div class="bg-gray-50 p-4 rounded-lg text-center text-gray-500">
                            No services available for this property.
                        </div>
                    `;
                }
            })
            .catch(error => {
                console.error('Error loading services:', error);
                servicesLoading.classList.add('hidden');
                servicesContainer.innerHTML = `
                    <div class="bg-red-50 p-4 rounded-lg text-center text-red-700">
                        Error loading services. Please try again.
                    </div>
                `;
            });
        }

        // Handle service checkbox changes
        document.addEventListener('change', function(e) {
            if (e.target && e.target.matches('.service-checkbox')) {
                const serviceId = e.target.value;
                const serviceItem = e.target.closest('.service-item');
                const priceInput = serviceItem.querySelector('.service-price');
                const quantityInput = serviceItem.querySelector('.service-quantity');
                const startDateInput = serviceItem.querySelector('.service-start-date');
                const totalInput = serviceItem.querySelector('.service-total');
                
                if (e.target.checked) {
                    priceInput.disabled = false;
                    quantityInput.disabled = false;
                    startDateInput.disabled = false;
                    
                    // Ensure quantity is at least 1
                    if (!quantityInput.value) {
                        quantityInput.value = 1;
                    }
                    
                    // Calculate initial total
                    calculateTotal(serviceId);
                } else {
                    priceInput.disabled = true;
                    quantityInput.disabled = true;
                    startDateInput.disabled = true;
                    totalInput.value = '0.00';
                }
            }
        });

        // Calculate total when quantity or price changes
        document.addEventListener('input', function(e) {
            if (e.target && (e.target.matches('.service-quantity') || e.target.matches('.service-price'))) {
                const serviceItem = e.target.closest('.service-item');
                const serviceId = serviceItem.id.replace('service-', '');
                calculateTotal(serviceId);
            }
        });

        // Function to calculate total
        function calculateTotal(serviceId) {
            const serviceItem = document.getElementById(`service-${serviceId}`);
            if (!serviceItem) return;
            
            const priceInput = serviceItem.querySelector('.service-price');
            const quantityInput = serviceItem.querySelector('.service-quantity');
            const totalInput = serviceItem.querySelector('.service-total');
            
            const price = parseFloat(priceInput.value) || 0;
            const quantity = parseInt(quantityInput.value) || 0;
            const total = (price * quantity).toFixed(2);
            
            totalInput.value = new Intl.NumberFormat('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(total);
        }

        // Handle unit selection change
        document.getElementById('UnitID').addEventListener('change', function() {
            const unitId = this.value;
            const selectedOption = this.options[this.selectedIndex];
            const propertyId = selectedOption.getAttribute('data-property-id');
            const wMeterNoField = document.getElementById('WMeterNo');
            const eMeterNoField = document.getElementById('EMeterNo');
            const meterInfoSection = document.getElementById('meterInformationSection');
            const servicesSection = document.getElementById('servicesSection');
            
            if (!unitId) {
                // Hide meter information and clear fields if no unit is selected
                meterInfoSection.classList.add('hidden');
                servicesSection.classList.add('hidden');
                resetMeterFields();
                return;
            }
            
            // Show meter information when a unit is selected
            meterInfoSection.classList.remove('hidden');
            
            // Load services for the selected property
            if (propertyId) {
                loadServices(propertyId);
            }

            // Show loading state
            const unitSelect = this;
            const originalValue = unitSelect.innerHTML;
            unitSelect.disabled = true;
            
            // Add loading class to the select
            unitSelect.classList.add('opacity-75', 'cursor-not-allowed');
            
            // Fetch unit details
            fetch(`/units/${unitId}/details`)
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Update meter fields with unit details and make meter numbers read-only
                        wMeterNoField.value = data.data.WMeterNo || '';
                        wMeterNoField.readOnly = true;
                        wMeterNoField.classList.add('bg-gray-100');
                        
                        // Only update initial reading if it's empty or hasn't been set yet
                        const wInitialReading = document.getElementById('WInitialReading');
                        if (!wInitialReading.value || wInitialReading.value === '0.000') {
                            wInitialReading.value = data.data.WLastReading || '0.000';
                        }
                        
                        eMeterNoField.value = data.data.EMeterNo || '';
                        eMeterNoField.readOnly = true;
                        eMeterNoField.classList.add('bg-gray-100');
                        
                        // Only update initial reading if it's empty or hasn't been set yet
                        const eInitialReading = document.getElementById('EInitialReading');
                        if (!eInitialReading.value || eInitialReading.value === '0.00') {
                            eInitialReading.value = data.data.ELastReading || '0.00';
                        }
                    } else {
                        resetMeterFields();
                        showToast('error', 'Failed to load unit details');
                    }
                })
                .catch(error => {
                    console.error('Error fetching unit details:', error);
                    resetMeterFields();
                    showToast('error', 'Error loading unit details. Please try again.');
                })
                .finally(() => {
                    // Reset select element
                    unitSelect.disabled = false;
                    unitSelect.classList.remove('opacity-75', 'cursor-not-allowed');
                });
        });

        function resetMeterFields() {
            document.getElementById('WMeterNo').value = '';
            document.getElementById('WInitialReading').value = '0.000';
            document.getElementById('EMeterNo').value = '';
            document.getElementById('EInitialReading').value = '0.00';
            
            // Make meter number fields editable when no unit is selected
            document.getElementById('WMeterNo').readOnly = false;
            document.getElementById('EMeterNo').readOnly = false;
            document.getElementById('WMeterNo').classList.remove('bg-gray-100');
            document.getElementById('EMeterNo').classList.remove('bg-gray-100');
        }

        function showToast(type, message) {
            // You can implement a toast notification here if you have one
            // For now, we'll just use alert
            alert(message);
        }

        // Auto-format phone number
        document.getElementById('TelNo').addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            
            // Format as 07XX XXX XXX or 01XX XXX XXX
            if (value.length > 3 && value.length <= 6) {
                value = value.replace(/(\d{4})(\d+)/, '$1 $2');
            } else if (value.length > 6) {
                value = value.replace(/(\d{4})(\d{3})(\d+)/, '$1 $2 $3');
            }
            
            e.target.value = value;
        });
        
        // Set deposit date to today if deposit amount is entered
        document.getElementById('Deposit').addEventListener('change', function(e) {
            const depositDate = document.getElementById('DepositDate');
            if (e.target.value && !depositDate.value) {
                depositDate.value = new Date().toISOString().split('T')[0];
            }
        });
        
        // Confirm before deleting
        document.getElementById('delete-button').addEventListener('click', function(e) {
            if (!confirm('Are you sure you want to delete this customer? This action cannot be undone.')) {
                e.preventDefault();
            }
        });
    </script>
    @endpush
@endsection
