@extends('layouts.app')

@push('styles')
<style>
    :root {
        --primary-blue: #0000FF;
        --accent-yellow: #FFFF00;
        --light-gray: #f8f9fa;
        --dark-gray: #343a40;
        --border-color: #dee2e6;
    }
    
    body {
        background-color: #f5f7fb;
        margin: 0;
        padding: 0;
    }
    
    .statement-container {
        background: #fff;
        box-shadow: none;
        border-radius: 0;
        margin: 0;
        padding: 0;
        overflow: hidden;
        border: 1px solid var(--border-color);
        padding: 0;
        margin: 0;
    }
    
    .company-header {
        background: linear-gradient(135deg, #ffffff 0%, #f8f9ff 100%);
        border-bottom: 2px solid var(--primary-blue);
        padding: 1rem 0;
        margin: 0;
        
    }
    
    .company-name {
        font-weight: 700;
        color: var(--primary-blue);
        margin-bottom: 0.5rem;
        font-size: 1.75rem;
        letter-spacing: -0.5px;
    }
    
    .statement-title {
        font-weight: 700;
        color: var(--dark-gray);
        font-size: 1.5rem;
        margin: 1rem 0;
        padding: 0;
        text-align: left;
        text-transform: uppercase;
        letter-spacing: 1px;
    }
    
    .section-title {
        font-size: 1rem;
        font-weight: 600;
        color: var(--primary-blue);
        border-bottom: 2px solid var(--primary-blue);
        padding-bottom: 0.5rem;
        margin-bottom: 1.25rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .customer-info-card, .summary-card {
        background: #fff;
        border-radius: 0;
        padding: 0.75rem;
        margin: 0;
        height: 100%;
        border: 1px solid var(--border-color);
        border-left: none;
        border-right: none;
    }
    
    .customer-info-card:hover, .summary-card:hover {
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.05);
    }
    
    .summary-card {
        border-left: 4px solid var(--primary-blue);
        background: linear-gradient(to right, #f8f9ff, #ffffff);
    }
    
    .transaction-table {
        width: 100%;
        margin: 0;
        border-collapse: collapse;
        background: #fff;
        border: none;
    }
    
    .transaction-table th {
        background-color: var(--primary-blue);
        color: white;
        font-weight: 500;
        text-transform: uppercase;
        font-size: 0.75rem;
        letter-spacing: 0.5px;
        padding: 0.5rem 0.75rem;
        border: none;
        text-align: left;
    }
    
    .transaction-table td {
        vertical-align: middle;
        padding: 0.5rem 0.75rem;
        border-bottom: 1px solid var(--border-color);
    }
    
    .transaction-table tbody tr:last-child td {
        border-bottom: none;
    }
    
    .transaction-table tbody tr:hover {
        background-color: rgba(0, 0, 255, 0.03);
    }
    
    .amount {
        font-family: 'Courier New', monospace;
        font-weight: 600;
    }
    
    .debit-amount {
        color: #dc3545;
    }
    
    .credit-amount {
        color: #28a745;
    }
    
    .balance-amount {
        font-weight: 700;
        color: var(--primary-blue);
    }
    
    .logo-container {
        background: white;
        padding: 0.75rem;
        border-radius: 0.5rem;
        box-shadow: 0 0.25rem 0.5rem rgba(0, 0, 0, 0.05);
        border: 1px solid var(--border-color);
    }
    
    .contact-info i {
        width: 20px;
        text-align: center;
        color: var(--primary-blue);
    }
    
    .btn-primary {
        background-color: var(--primary-blue);
        border-color: var(--primary-blue);
    }
    
    .btn-primary:hover {
        background-color: #0000cc;
        border-color: #0000cc;
    }
    
    .form-control:focus, .form-select:focus {
        border-color: var(--primary-blue);
        box-shadow: 0 0 0 0.25rem rgba(0, 0, 255, 0.15);
    }
    
    .breadcrumb {
        margin: 0 0 0.5rem 0;
        background: transparent;
        padding: 0.5rem 1rem;
    }
    
    .breadcrumb-item a {
        color: var(--primary-blue);
        text-decoration: none;
    }
    
    .breadcrumb-item.active {
        color: #6c757d;
    }
</style>
@endpush

@section('content')
<div class="container-fluid p-0">
    <div class="row g-0">
        <div class="col-12">
            <nav aria-label="breadcrumb" class="mb-4">
                <ol class="breadcrumb bg-transparent p-0">
                    <li class="breadcrumb-item"><a href="{{ route('customers.index') }}" class="text-primary">Customers</a></li>
                    <li class="breadcrumb-item"><a href="{{ route('customers.show', $customer) }}" class="text-primary">{{ $customer->CustomerName }}</a></li>
                    <li class="breadcrumb-item active" aria-current="page">Account Statement</li>
                </ol>
            </nav>

            <div class="statement-container">
                <!-- Company Header -->
                <div class="company-header">
                    <div class=" px-4 d-flex justify-content-between align-items-start">
                        <!-- Company Info - Left Side -->
                        <div class="company-details">
                            <h1 class="company-name">{{ $company['name'] ?? 'Company Name' }}</h1>
                            @if(isset($company['address']) && $company['address'])
                                <p class="text-muted mb-2">{{ $company['address'] }}</p>
                            @endif
                            <div class="contact-info">
                                @if(isset($company['phone']) && $company['phone'])
                                    <div class="d-flex align-items-center mb-1">
                                        <i class="fas fa-phone-alt me-2 text-primary"></i>
                                        <span class="text-muted">{{ $company['phone'] }}</span>
                                    </div>
                                @endif
                                @if(isset($company['email']) && $company['email'])
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-envelope me-2 text-primary"></i>
                                        <span class="text-muted">{{ $company['email'] }}</span>
                                    </div>
                                @endif
                            </div>
                        </div>
                        
                        <!-- Logo - Right Side -->
                        @if(isset($company['logo']) && $company['logo'])
                            <div class="logo-container">
                                <img src="{{ $company['logo'] }}" alt="{{ $company['name'] }}" style="max-height: 80px; max-width: 200px; object-fit: contain;">
                            </div>
                        @endif
                    </div>
                </div>
                
                <!-- Statement Header -->
                <div class="p-4 border-bottom">
                    <div class="d-flex justify-content-between align-items-center flex-wrap gap-3">
                        <h2 class="statement-title mb-0">Account Statement</h2>
                        
                        <!-- Date Range Selector -->
                        <div class="d-flex align-items-center gap-2">
                            <form method="GET" class="d-flex gap-2">
                                <div class="input-group input-group-sm">
                                    <select name="range" id="date-range" class="form-select form-select-sm" style="min-width: 180px;">
                                        @foreach($dateRanges as $value => $label)
                                            <option value="{{ $value }}" {{ $selectedRange === $value ? 'selected' : '' }}>{{ $label }}</option>
                                        @endforeach
                                    </select>
                                    <div id="custom-date-range" class="input-group {{ $selectedRange !== 'custom' ? 'd-none' : '' }}">
                                        <input type="date" name="start_date" class="form-control form-control-sm" 
                                               value="{{ $startDate->format('Y-m-d') }}">
                                        <input type="date" name="end_date" class="form-control form-control-sm" 
                                               value="{{ $endDate->format('Y-m-d') }}">
                                        <button type="submit" class="btn btn-sm btn-outline-primary">
                                            <i class="fas fa-check"></i>
                                        </button>
                                    </div>
                                </div>
                            </form>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="d-flex gap-2">
                            <form action="{{ route('customers.statement.email', $customer) }}" method="POST" class="d-inline">
                                @csrf
                                <input type="hidden" name="start_date" value="{{ $startDate->format('Y-m-d') }}">
                                <input type="hidden" name="end_date" value="{{ $endDate->format('Y-m-d') }}">
                                <button type="submit" class="btn btn-sm btn-success" 
                                        onclick="return confirm('Send statement to {{ $customer->Email }}?')">
                                    <i class="fas fa-envelope me-1"></i> Email
                                </button>
                            </form>
                            <a href="{{ route('customers.statement.export', array_merge(request()->query(), ['customer' => $customer])) }}" 
                               class="btn btn-sm btn-primary">
                                <i class="fas fa-file-pdf me-1"></i> Export
                            </a>
                        </div>
                    </div>

                    <!-- Customer Info & Summary -->
                    <div class="row g-4 mb-4">
                        <!-- Customer Info -->
                        <div class="col-md-6">
                            <div class="customer-info-card">
                                <h6 class="section-title">Customer Information</h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>Name:</strong> {{ $customer->CustomerName }}</p>
                                        <p class="mb-1"><strong>Account #:</strong> {{ $customer->AccountNo }}</p>
                                        @if($customer->Email)
                                            <p class="mb-1"><strong>Email:</strong> {{ $customer->Email }}</p>
                                        @endif
                                    </div>
                                    <div class="col-md-6">
                                        <p class="mb-1"><strong>Address:</strong></p>
                                        <p class="mb-1">{{ $customer->Address1 }}</p>
                                        @if($customer->Address2)
                                            <p class="mb-1">{{ $customer->Address2 }}</p>
                                        @endif
                                        <p class="mb-0"><strong>Phone:</strong> {{ $customer->TelNo }}</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Statement Summary -->
                        <div class="col-md-6">
                            <div class="summary-card">
                                <h6 class="section-title">Statement Summary</h6>
                                <div class="table-responsive">
                                    <table class="table table-sm table-borderless mb-0">
                                        <tr>
                                            <td>Statement Period:</td>
                                            <td class="text-end fw-medium">
                                                {{ $startDate->format('F j, Y') }} - {{ $endDate->format('F j, Y') }}
                                            </td>
                                        </tr>
                                        <tr>
                                            <td>Opening Balance:</td>
                                            <td class="text-end fw-medium">
                                                {{ number_format($statement['summary']['opening_balance'], 2) }}
                                            </td>
                                        </tr>
                                        <tr class="border-top">
                                            <td>Total Debits:</td>
                                            <td class="text-end fw-medium text-danger">
                                                {{ number_format($statement['summary']['total_debits'], 2) }}
                                            </td>
                                        </tr>
                                        <tr>
                                            <td>Total Credits:</td>
                                            <td class="text-end fw-medium text-success">
                                                {{ number_format($statement['summary']['total_credits'], 2) }}
                                            </td>
                                        </tr>
                                        <tr class="border-top">
                                            <td class="fw-bold">Closing Balance:</td>
                                            <td class="text-end fw-bold">
                                                {{ number_format($statement['summary']['closing_balance'], 2) }}
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Transactions Table -->
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="thead-light">
                                <tr>
                                    <th>Date</th>
                                    <th>Description</th>
                                    <th>Reference</th>
                                    <th class="text-right">Debit</th>
                                    <th class="text-right">Credit</th>
                                    <th class="text-right">Balance</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($statement['transactions'] as $transaction)
                                    <tr class="{{ $transaction['type'] === 'opening_balance' ? 'table-info' : '' }}">
                                        <td>{{ \Carbon\Carbon::parse($transaction['date'])->format('M j, Y') }}</td>
                                        <td>
                                            {{ $transaction['description'] }}
                                            @if(isset($transaction['entity']->BillType) && $transaction['entity']->BillType)
                                                <br><small class="text-muted">{{ $transaction['entity']->BillType }}</small>
                                            @endif
                                        </td>
                                        <td>
                                            @if($transaction['type'] === 'bill')
                                                @php
                                                    $billType = $transaction['bill_type'] ?? strtolower($transaction['entity']->BillType ?? '');
                                                    $routeName = '';
                                                    
                                                    if ($billType === 'rent') {
                                                        $routeName = 'bills.rent.show';
                                                    } else {
                                                        $routeName = 'bills.utility.show';
                                                    }
                                                @endphp
                                                
                                                @if(Route::has($routeName))
                                                    <a href="{{ route($routeName, $transaction['entity']->BillID) }}" class="text-primary">
                                                        {{ $transaction['reference'] }}
                                                    </a>
                                                @else
                                                    {{ $transaction['reference'] }}
                                                @endif
                                            @elseif($transaction['type'] === 'payment' && Route::has('payments.show'))
                                                <a href="{{ route('payments.show', $transaction['entity']->PaymentID) }}" class="text-success">
                                                    {{ $transaction['reference'] }}
                                                </a>
                                            @else
                                                {{ $transaction['reference'] }}
                                            @endif
                                        </td>
                                        <td class="text-right">
                                            @if($transaction['debit'] > 0)
                                                {{ number_format($transaction['debit'], 2) }}
                                            @endif
                                        </td>
                                        <td class="text-right">
                                            @if($transaction['credit'] > 0)
                                                {{ number_format($transaction['credit'], 2) }}
                                            @endif
                                        </td>
                                        <td class="text-right font-weight-bold">
                                            {{ number_format($transaction['balance'], 2) }}
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const dateRangeSelect = document.getElementById('date-range');
        const customDateRange = document.getElementById('custom-date-range');
        
        dateRangeSelect.addEventListener('change', function() {
            if (this.value === 'custom') {
                customDateRange.classList.remove('d-none');
            } else {
                customDateRange.classList.add('d-none');
                // Auto-submit when a predefined range is selected
                this.form.submit();
            }
        });
        
        // Initialize date pickers
        $('input[type="date"]').on('change', function() {
            if (dateRangeSelect.value === 'custom') {
                dateRangeSelect.form.submit();
            }
        });
    });
</script>
@endpush
