@extends('layouts.app')

@section('content')
@push('styles')
<style>
    /* Tooltip styles */
    [data-tooltip] {
        position: relative;
    }
    
    [data-tooltip] .tooltip-text {
        visibility: hidden;
        width: 100px;
        background-color: #1E40AF;
        color: #fff;
        text-align: center;
        border-radius: 4px;
        padding: 4px 8px;
        position: absolute;
        z-index: 1;
        bottom: 125%;
        left: 50%;
        transform: translateX(-50%);
        opacity: 0;
        transition: opacity 0.2s, visibility 0.2s;
        font-size: 12px;
        white-space: nowrap;
        pointer-events: none;
    }
    
    [data-tooltip]:hover .tooltip-text {
        visibility: visible;
        opacity: 1;
    }
    
    [data-tooltip] .tooltip-text::after {
        content: "";
        position: absolute;
        top: 100%;
        left: 50%;
        margin-left: -5px;
        border-width: 5px;
        border-style: solid;
        border-color: #1E40AF transparent transparent transparent;
    }
</style>
@endpush
@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const tabLinks = document.querySelectorAll('.tab-link');
        const unitRows = document.querySelectorAll('.unit-row');
        
        // Function to filter units based on tab
        function filterUnits(tab) {
            // Hide all unit rows first
            unitRows.forEach(row => row.classList.add('hidden'));
            
            // Hide all empty states first
            document.querySelectorAll('.empty-state').forEach(state => state.classList.add('hidden'));
            
            // Count visible units for this tab
            let visibleCount = 0;
            
            // Show/hide units based on tab
            unitRows.forEach(row => {
                const status = row.getAttribute('data-status');
                const isOccupied = status === 'occupied';
                const isVacant = status === 'vacant';
                
                const shouldShow = tab === 'all' || 
                                 (tab === 'occupied' && isOccupied) || 
                                 (tab === 'vacant' && isVacant);
                
                if (shouldShow) {
                    row.classList.remove('hidden');
                    visibleCount++;
                }
            });
            
            // Always show the appropriate empty state for the current tab
            const emptyState = document.querySelector(`.empty-state.${tab}-state`);
            if (emptyState) {
                emptyState.classList.remove('hidden');
            }
            
            // Update active tab with Sulis color scheme
            tabLinks.forEach(link => {
                const linkTab = link.getAttribute('data-tab');
                const isActive = linkTab === tab;
                
                if (isActive) {
                    link.classList.add('border-yellow-400', 'text-blue-600', 'font-semibold');
                    link.classList.remove('text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
                    link.style.borderBottomColor = '#FFFF00'; // Sulis yellow
                    link.style.borderBottomWidth = '3px';
                    link.style.color = '#0000FF'; // Sulis blue
                } else {
                    link.classList.remove('border-yellow-400', 'text-blue-600', 'font-semibold');
                    link.classList.add('text-gray-500', 'hover:text-blue-600');
                    link.style.borderBottomColor = '';
                    link.style.borderBottomWidth = '';
                    link.style.color = '';
                }
            });
        }
        
        // Add click event listeners to tabs
        tabLinks.forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                const tab = this.getAttribute('data-tab');
                filterUnits(tab);
                
                // Update URL hash without scrolling
                history.pushState(null, null, '#' + tab);
            });
            
            // Ensure all tabs are clickable
            link.style.pointerEvents = 'auto';
            link.style.cursor = 'pointer';
        });
        
        // Check URL hash on page load
        const hash = window.location.hash.substring(1);
        if (['all', 'occupied', 'vacant'].includes(hash)) {
            filterUnits(hash);
        } else {
            // Default to 'all' tab
            filterUnits('all');
        }
    });
</script>
@endpush
<div class="py-6">
    <div class="max-w-7xl mx-auto sm:px-6 lg:px-8">
        <!-- Property Header -->
        <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg mb-6">
            <div class="p-6 bg-white border-b border-gray-200">
                <div class="flex items-center mb-4">
                    <a href="{{ route('properties.index') }}" class="inline-flex items-center text-sm font-medium text-gray-500 hover:text-gray-700">
                        <svg class="h-5 w-5 mr-1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                            <path fill-rule="evenodd" d="M9.707 16.707a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414l6-6a1 1 0 011.414 1.414L5.414 9H17a1 1 0 110 2H5.414l4.293 4.293a1 1 0 010 1.414z" clip-rule="evenodd" />
                        </svg>
                        Back to Properties
                    </a>
                </div>
                <div class="md:flex md:items-center md:justify-between">
                    <div class="flex-1 min-w-0">
                        <h2 class="text-2xl font-bold leading-7 text-gray-900 sm:text-3xl sm:truncate">
                            {{ $property->PropName }}
                        </h2>
                        <div class="mt-1 flex flex-col sm:flex-row sm:flex-wrap sm:mt-0 sm:space-x-6">
                            <div class="mt-2 flex items-center text-sm text-gray-500">
                                <svg class="flex-shrink-0 mr-1.5 h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fill-rule="evenodd" d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z" clip-rule="evenodd" />
                                </svg>
                                {{ $property->City }}
                            </div>
                            <div class="mt-2 flex items-center text-sm text-gray-500">
                                <svg class="flex-shrink-0 mr-1.5 h-5 w-5 text-gray-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fill-rule="evenodd" d="M4 4a2 2 0 012-2h8a2 2 0 012 2v12a2 2 0 002 2h-2a2 2 0 01-2 2H6a2 2 0 01-2-2V6a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2H6a2 2 0 01-2-2V4z" clip-rule="evenodd" />
                                </svg>
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                                    {{ $property->PropertyType }}
                                </span>
                            </div>
                        </div>
                    </div>
                     
                    <div class="mt-4 flex md:mt-0 md:ml-4 space-x-3">
                        <a href="{{ route('properties.edit', $property->PropID) }}" 
                           class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500">
                            <svg class="-ml-1 mr-2 h-5 w-5 text-gray-500" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                <path d="M13.586 3.586a2 2 0 112.828 2.828l-.793.793-2.828-2.828.793-.793zM11.379 5.793L3 14.172V17h2.828l8.38-8.379-2.83-2.828z" />
                            </svg>
                           Edit Property
                        </a>
                        <a href="{{ route('properties.units.create', $property->PropID) }}" 
                           class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                            <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                <path fill-rule="evenodd" d="M10 3a1 1 0 011 1v5h5a1 1 0 110 2h-5v5a1 1 0 11-2 0v-5H4a1 1 0 110-2h5V4a1 1 0 011-1z" clip-rule="evenodd" />
                            </svg>
                            Add Unit
                        </a>
                      
                        <!-- <form action="{{ route('properties.destroy', $property->PropID) }}" method="POST" onsubmit="return confirm('Are you sure you want to delete this property?');">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500">
                                <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fill-rule="evenodd" d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z" clip-rule="evenodd" />
                                </svg>
                                Delete
                            </button>
                        </form> -->
                    </div>
                </div>
            </div>
        </div>

        <!-- Stats -->
        <div class="grid grid-cols-1 gap-5 sm:grid-cols-2 lg:grid-cols-4 mb-6">
            <!-- Total Units Card -->
            <div class="bg-white overflow-hidden shadow rounded-lg transform transition-all duration-300 hover:shadow-lg hover:-translate-y-1">
                <div class="px-4 py-5 sm:p-6">
                    <div class="flex items-center">
                        <div class="flex-shrink-0 bg-indigo-100 rounded-md p-3">
                            <svg class="h-6 w-6 text-indigo-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                            </svg>
                        </div>
                        <div class="ml-5 w-0 flex-1">
                            <dl>
                                <dt class="text-sm font-medium text-gray-500 truncate">Total Units</dt>
                                <dd class="flex items-baseline">
                                    <div class="text-2xl font-semibold text-gray-900">{{ $stats['total_units'] ?? 0 }}</div>
                                    @if(isset($stats['total_units_prev']) && $stats['total_units_prev'] > 0)
                                        @php
                                            $change = (($stats['total_units'] - $stats['total_units_prev']) / $stats['total_units_prev']) * 100;
                                            $isIncrease = $change >= 0;
                                        @endphp
                                        <div class="ml-2 flex items-baseline text-sm font-semibold {{ $isIncrease ? 'text-green-600' : 'text-red-600' }}">
                                            {{ $isIncrease ? '↑' : '↓' }} {{ number_format(abs($change), 0) }}%
                                            <span class="sr-only">{{ $isIncrease ? 'Increased' : 'Decreased' }} by</span>
                                        </div>
                                    @endif
                                </dd>
                            </dl>
                        </div>
                    </div>
                </div>
               
            </div>

            <!-- Other Stats Cards -->
           

            <div class="bg-white overflow-hidden shadow rounded-lg transform transition-all duration-300 hover:shadow-lg hover:-translate-y-1">
                <div class="px-4 py-5 sm:p-6">
                    <div class="flex items-center">
                        <div class="flex-shrink-0 bg-green-100 rounded-md p-3">
                            <svg class="h-6 w-6 text-green-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z" />
                            </svg>
                        </div>
                        <div class="ml-5 w-0 flex-1">
                            <dl>
                                <dt class="text-sm font-medium text-gray-500 truncate">Occupied Units</dt>
                                <dd class="flex items-baseline">
                                    <div class="text-2xl font-semibold text-gray-900">{{ $stats['occupied_units'] ?? 0 }}</div>
                                </dd>
                            </dl>
                        </div>
                    </div>
                </div>
              
            </div>

            <div class="bg-white overflow-hidden shadow rounded-lg">
                <div class="px-4 py-5 sm:p-6">
                    <div class="flex items-center">
                        <div class="flex-shrink-0 bg-yellow-500 rounded-md p-3">
                            <svg class="h-6 w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
                            </svg>
                        </div>
                        <div class="ml-5 w-0 flex-1">
                            <dl>
                                <dt class="text-sm font-medium text-gray-500 truncate">Vacant Units</dt>
                                <dd class="flex items-baseline">
                                    <div class="text-2xl font-semibold text-gray-900">{{ $stats['vacant_units'] ?? 0 }}</div>
                                </dd>
                            </dl>
                        </div>
                    </div>
                </div>
                
            </div>

            <div class="bg-white overflow-hidden shadow rounded-lg">
                <div class="px-4 py-5 sm:p-6">
                    <div class="flex items-center">
                        <div class="flex-shrink-0 bg-red-500 rounded-md p-3">
                            <svg class="h-6 w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                            </svg>
                        </div>
                        <div class="ml-5 w-0 flex-1">
                            <dl>
                                <dt class="text-sm font-medium text-gray-500 truncate">Services</dt>
                                <dd class="flex items-baseline">
                                    <div class="text-2xl font-semibold text-gray-900">{{ $services_count ?? 0 }}</div>
                                    
                                </dd>
                            </dl>
                        </div>
                    </div>
                </div>
              
            </div>
        </div>

        <!-- Property Information -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
            <div class="bg-white shadow-lg overflow-hidden sm:rounded-lg border border-gray-100 hover:shadow-xl transition-shadow duration-300">
                <div class="px-6 py-4 bg-gradient-to-r from-blue-600 to-blue-700">
                    <div class="flex items-center">
                        <svg class="h-6 w-6 text-yellow-400 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5m-1-8h1m-1 4h1m4-4h1m-1 4h1m4-4h1m-1 4h1" />
                        </svg>
                        <h3 class="text-lg font-semibold text-white">Property Information</h3>
                    </div>
                    <p class="mt-1 text-sm text-blue-100">Detailed information about the property</p>
                </div>
                <div class="border-t border-gray-200 px-4 py-5 sm:p-0">
                    <dl class="sm:divide-y sm:divide-gray-200">
                        <div class="py-4 sm:py-5 sm:grid sm:grid-cols-3 sm:gap-4 sm:px-6">
                            <dt class="text-sm font-medium text-gray-500">Description</dt>
                            <dd class="mt-1 text-sm text-gray-900 sm:mt-0 sm:col-span-2">
                                {{ $property->PropDesc ?? 'No description provided.' }}
                            </dd>
                        </div>
                        <div class="py-4 sm:py-5 sm:grid sm:grid-cols-3 sm:gap-4 sm:px-6">
                            <dt class="text-sm font-medium text-gray-500">Months Per Year</dt>
                            <dd class="mt-1 text-sm text-gray-900 sm:mt-0 sm:col-span-2">
                                {{ $property->NoMonthPerYear ?? '12' }}
                            </dd>
                        </div>
                        <div class="py-4 sm:py-5 sm:grid sm:grid-cols-3 sm:gap-4 sm:px-6">
                            <dt class="text-sm font-medium text-gray-500">Meter Information</dt>
                            <dd class="mt-1 text-sm text-gray-900 sm:mt-0 sm:col-span-2">
                                <div class="grid grid-cols-1 gap-2">
                                    <div class="flex items-center">
                                        <svg class="h-5 w-5 text-gray-400 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z" />
                                        </svg>
                                        <span class="font-medium">Electric:</span> 
                                        <span class="ml-2">{{ $property->EMeterNo ?? 'Not specified' }}</span>
                                    </div>
                                    <div class="flex items-center">
                                        <svg class="h-5 w-5 text-blue-400 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4" />
                                        </svg>
                                        <span class="font-medium">Water:</span> 
                                        <span class="ml-2">{{ $property->WMeterNo ?? 'Not specified' }}</span>
                                    </div>
                                </div>
                            </dd>
                        </div>
                    </dl>
                </div>
            </div>

            <!-- Property Services -->
            <div class="bg-white shadow-lg overflow-hidden sm:rounded-lg border border-gray-100 hover:shadow-xl transition-shadow duration-300">
                <div class="px-6 py-4 bg-gradient-to-r from-yellow-500 to-yellow-600">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center">
                            <svg class="h-6 w-6 text-white mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z" />
                            </svg>
                            <h3 class="text-lg font-semibold text-white">Property Services</h3>
                        </div>
                        <a href="{{ route('prop-services.create') }}?property_id={{ $property->PropID }}" 
                           class="inline-flex items-center px-3 py-1.5 border border-transparent text-xs font-medium rounded-md shadow-sm text-yellow-700 bg-white hover:bg-yellow-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-500 transition-colors duration-200">
                            <svg class="-ml-0.5 mr-1.5 h-4 w-4" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                <path fill-rule="evenodd" d="M10 3a1 1 0 011 1v5h5a1 1 0 110 2h-5v5a1 1 0 11-2 0v-5H4a1 1 0 110-2h5V4a1 1 0 011-1z" clip-rule="evenodd" />
                            </svg>
                            Add Service
                        </a>
                    </div>
                    <p class="mt-1 text-sm text-yellow-100">Common services and utilities for this property</p>
                </div>
                <div class="bg-white overflow-hidden">
                    @if($services->count() > 0)
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-blue-50">
                                    <tr>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-blue-700 uppercase tracking-wider">Service Name</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-blue-700 uppercase tracking-wider">Description</th>
                                        <th scope="col" class="px-6 py-3 text-right text-xs font-medium text-blue-700 uppercase tracking-wider">Price</th>
                                        <th scope="col" class="px-6 py-3 text-center text-xs font-medium text-blue-700 uppercase tracking-wider">Billing Period</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white divide-y divide-gray-100">
                                    @foreach($services as $service)
                                        <tr class="hover:bg-blue-50 transition-colors duration-150">
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <a href="{{ route('prop-services.show', $service->IDService) }}" class="flex items-center group">
                                                    <div class="flex-shrink-0 h-8 w-8 bg-blue-100 rounded-md flex items-center justify-center text-blue-600 mr-3">
                                                        <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z" />
                                                        </svg>
                                                    </div>
                                                    <span class="text-sm font-medium text-blue-700 group-hover:text-blue-900">
                                                        {{ $service->ServiceName }}
                                                    </span>
                                                </a>
                                            </td>
                                            <td class="px-6 py-4 text-sm text-gray-600">
                                                <div class="line-clamp-2">
                                                    {{ $service->Description ?? 'No description provided' }}
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-right">
                                                <span class="text-sm font-medium text-gray-900">
                                                {{ number_format($service->Price, 2) }}
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-center text-gray-500">
                                                @php
                                                    $periods = [
                                                        'Monthly' => 'Month',
                                                        'Quarterly' => 'Quarter',
                                                        'Annually' => 'Year'
                                                    ];
                                                    $period = $periods[$service->NoMonthPerYear] ?? $service->NoMonthPerYear;
                                                @endphp
                                                {{ $period }}
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <div class="text-center py-4">
                            <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                            <h3 class="mt-2 text-sm font-medium text-gray-900">No services</h3>
                            <p class="mt-1 text-sm text-gray-500">Get started by adding a new service.</p>
                            <div class="mt-6">
                                <a href="{{ route('prop-services.create') }}?property_id={{ $property->PropID }}" class="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                    <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                        <path fill-rule="evenodd" d="M10 3a1 1 0 011 1v5h5a1 1 0 110 2h-5v5a1 1 0 11-2 0v-5H4a1 1 0 110-2h5V4a1 1 0 011-1z" clip-rule="evenodd" />
                                    </svg>
                                    New Service
                                </a>
                            </div>
                        </div>
                    @endif
                </div>
                </div>
        
            </div>
        </div>

      

        <!-- Units Section -->
        <div class="bg-white shadow overflow-hidden sm:rounded-lg mt-6 mx-4 sm:mx-6 lg:mx-8" id="units">
            <div class="px-4 py-2 sm:px-6 border-b border-gray-200">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-lg leading-6 font-medium text-gray-900">Units</h3>
                        <p class="mt-1 max-w-2xl text-sm text-gray-500">List of units in this property.</p>
                    </div>
                    <div>
                        <a href="{{ route('properties.units.index', $property->PropID) }}" class="text-sm text-indigo-600 hover:text-indigo-900">
                            View all units management
                        </a>
                    </div>
                </div>
                
                <!-- Tabs -->
                <div class="mt-6 border-b border-gray-200">
                    <nav class="-mb-px flex space-x-1" aria-label="Tabs">
                        <a href="#" data-tab="all" class="tab-link border-b-2 border-transparent text-gray-500 hover:text-blue-600 hover:border-gray-300 whitespace-nowrap py-3 px-4 font-medium text-sm transition-all duration-200 ease-in-out active">
                            <div class="flex items-center">
                                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5m-1-8h1m-1 4h1m4-4h1m-1 4h1m4-4h1m-1 4h1"></path>
                                </svg>
                                <span>All Units</span>
                                <span class="bg-blue-100 text-blue-800 ml-2 py-0.5 px-2 rounded-full text-xs font-medium">{{ $units->total() }}</span>
                            </div>
                        </a>
                        <a href="#" data-tab="occupied" class="tab-link border-b-2 border-transparent text-gray-500 hover:text-blue-600 hover:border-gray-300 whitespace-nowrap py-3 px-4 font-medium text-sm transition-all duration-200 ease-in-out">
                            <div class="flex items-center">
                                <svg class="w-4 h-4 mr-2 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                </svg>
                                <span>Occupied</span>
                                <span class="bg-green-100 text-green-800 ml-2 py-0.5 px-2 rounded-full text-xs font-medium">{{ $occupied_units_count ?? 0 }}</span>
                            </div>
                        </a>
                        <a href="#" data-tab="vacant" class="tab-link border-b-2 border-transparent text-gray-500 hover:text-blue-600 hover:border-gray-300 whitespace-nowrap py-3 px-4 font-medium text-sm transition-all duration-200 ease-in-out">
                            <div class="flex items-center">
                                <svg class="w-4 h-4 mr-2 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                                </svg>
                                <span>Vacant</span>
                                <span class="bg-blue-100 text-blue-800 ml-2 py-0.5 px-2 rounded-full text-xs font-medium">{{ $vacant_units_count ?? 0 }}</span>
                            </div>
                        </a>
                    </nav>
                </div>
            </div>
            <div class="bg-white shadow overflow-hidden sm:rounded-lg">
                <div class="flex flex-col">
                    <div class="-my-2 overflow-x-auto sm:-mx-6 lg:-mx-8">
                        <div class="py-2 align-middle inline-block min-w-full sm:px-6 lg:px-8">
                            <div class="shadow overflow-hidden border-b border-gray-200 sm:rounded-lg">
                                <table class="min-w-full divide-y divide-gray-200 rounded-lg overflow-hidden">
                                    <thead class="bg-blue-600">
                                        <tr>
                                            <th scope="col" class="px-6 py-4 text-left text-xs font-semibold text-white uppercase tracking-wider">
                                                <div class="flex items-center">
                                                    <span>Unit ID</span>
                                                    <svg class="ml-1 w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16V4m0 0L3 8m4-4l4 4m6 0v12m0 0l4-4m-4 4l-4-4"></path>
                                                    </svg>
                                                </div>
                                            </th>
                                            <th scope="col" class="px-6 py-4 text-left text-xs font-semibold text-white uppercase tracking-wider">
                                                Unit Identity
                                            </th>
                                            <th scope="col" class="px-6 py-4 text-left text-xs font-semibold text-white uppercase tracking-wider">
                                                Status
                                            </th>
                                            <th scope="col" class="px-6 py-4 text-left text-xs font-semibold text-white uppercase tracking-wider">
                                                <div class="flex items-center">
                                                    <span>Monthly Rent</span>
                                                    <svg class="ml-1 w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16V4m0 0L3 8m4-4l4 4m6 0v12m0 0l4-4m-4 4l-4-4"></path>
                                                    </svg>
                                                </div>
                                            </th>
                                            <th scope="col" class="px-6 py-4 text-left text-xs font-semibold text-white uppercase tracking-wider">
                                                Tenants
                                            </th>
                                            <th scope="col" class="px-6 py-4 text-right text-xs font-semibold text-white uppercase tracking-wider">
                                                Actions
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody class="bg-white divide-y divide-gray-100">
                                        @forelse($units as $unit)
                                            @php
                                                $isOccupied = $unit->customers_count > 0;
                                            @endphp
                                            <tr class="unit-row bg-white hover:bg-blue-50 transition-colors duration-150" 
                                                data-status="{{ $isOccupied ? 'occupied' : 'vacant' }}">
                                                <td class="px-6 py-4 whitespace-nowrap">
                                                    <div class="text-sm font-semibold text-blue-900">
                                                        #{{ $unit->UnitID }}
                                                    </div>
                                                </td>
                                                <td class="px-6 py-4 whitespace-nowrap">
                                                    <div class="flex items-center">
                                                        <div class="flex-shrink-0 h-10 w-10 flex items-center justify-center rounded-lg bg-blue-100 text-blue-600 mr-3">
                                                            <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5m-1-8h1m-1 4h1m4-4h1m-1 4h1m4-4h1m-1 4h1"></path>
                                                            </svg>
                                                        </div>
                                                        <div>
                                                            <div class="text-sm font-medium text-gray-900">{{ $unit->UnitIdentity }}</div>
                                                            <div class="text-xs text-gray-500">
                                                                @if($unit->BlockNo || $unit->FloorNo || $unit->DoorNo)
                                                                    {{ $unit->BlockNo ? 'Block ' . $unit->BlockNo : '' }}
                                                                    {{ $unit->FloorNo ? '• Floor ' . $unit->FloorNo : '' }}
                                                                    {{ $unit->DoorNo ? '• Door ' . $unit->DoorNo : '' }}
                                                                @else
                                                                    No location details
                                                                @endif
                                                            </div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td class="px-6 py-4 whitespace-nowrap">
                                                    @if($isOccupied)
                                                        <span class="px-2.5 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                                            <span class="h-2 w-2 rounded-full bg-green-500 mr-1.5 mt-1"></span>
                                                            Occupied
                                                        </span>
                                                    @else
                                                        <span class="px-2.5 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                                                            <span class="h-2 w-2 rounded-full bg-blue-500 mr-1.5 mt-1"></span>
                                                            Vacant
                                                        </span>
                                                    @endif
                                                </td>
                                                <td class="px-6 py-4 whitespace-nowrap">
                                                    <div class="text-sm font-medium text-gray-900">Ksh {{ number_format($unit->MonthlyRent, 2) }}</div>
                                                    <div class="text-xs text-gray-500">per month</div>
                                                </td>
                                                <td class="px-6 py-4 whitespace-nowrap">
                                                    <div class="flex items-center">
                                                        <span class="h-8 w-8 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 font-medium text-sm mr-3">
                                                            {{ $unit->customers_count ?? 0 }}
                                                        </span>
                                                        <span class="text-sm text-gray-700">
                                                            {{ $unit->customers_count == 1 ? 'Tenant' : 'Tenants' }}
                                                        </span>
                                                    </div>
                                                </td>
                                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                    <div class="flex items-center justify-end space-x-2">
                                                        <a href="{{ route('properties.units.assignments.index', ['property' => $property->PropID, 'unit' => $unit->UnitID]) }}" 
                                                           class="text-blue-600 hover:text-blue-800 p-2 rounded-full hover:bg-blue-50 transition-colors duration-200 group relative"
                                                           data-tooltip="View Details">
                                                            <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                                                            </svg>
                                                            <span class="tooltip-text">View Details</span>
                                                        </a>
                                                        <a href="{{ route('properties.units.edit', ['property' => $property->PropID, 'unit' => $unit->UnitID]) }}" 
                                                           class="text-yellow-600 hover:text-yellow-800 p-2 rounded-full hover:bg-yellow-50 transition-colors duration-200 group relative"
                                                           data-tooltip="Edit Unit">
                                                            <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                                                            </svg>
                                                            <span class="tooltip-text">Edit Unit</span>
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        @empty
                                            <!-- Empty state for All Units tab -->
                                            <tr class="empty-state all-units-state">
                                                <td colspan="6" class="px-6 py-16">
                                                    <div class="text-center bg-blue-50 rounded-lg p-8 max-w-2xl mx-auto">
                                                        <div class="inline-flex items-center justify-center h-16 w-16 rounded-full bg-blue-100 mb-4">
                                                            <svg class="h-8 w-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5m-1-8h1m-1 4h1m4-4h1m-1 4h1m4-4h1m-1 4h1"></path>
                                                            </svg>
                                                        </div>
                                                        <h3 class="text-lg font-medium text-gray-900">No Units Added Yet</h3>
                                                        <p class="mt-2 text-sm text-gray-500">
                                                            You haven't added any units to this property. Get started by adding your first unit to manage tenants and track payments.
                                                        </p>
                                                        <div class="mt-6">
                                                            <a href="{{ route('properties.units.create', $property->PropID) }}" class="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                                                <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                    <path fill-rule="evenodd" d="M10 3a1 1 0 011 1v5h5a1 1 0 110 2h-5v5a1 1 0 11-2 0v-5H4a1 1 0 110-2h5V4a1 1 0 011-1z" clip-rule="evenodd" />
                                                                </svg>
                                                                Add First Unit
                                                            </a>
                                                        </div>
                                                    </div>
                                                </td>
                                            </tr>
                                            
                                            <!-- Empty state for Occupied tab -->
                                            <tr class="empty-state occupied-state hidden">
                                                <td colspan="6" class="px-6 py-16">
                                                    <div class="text-center bg-yellow-50 rounded-lg p-8 max-w-2xl mx-auto">
                                                        <div class="inline-flex items-center justify-center h-16 w-16 rounded-full bg-yellow-100 mb-4">
                                                            <svg class="h-8 w-8 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                                                            </svg>
                                                        </div>
                                                        <h3 class="text-lg font-medium text-gray-900">No Occupied Units</h3>
                                                        <p class="mt-2 text-sm text-gray-500">
                                                            There are currently no occupied units in this property. When you assign tenants to units, they will appear here.
                                                        </p>
                                                    </div>
                                                </td>
                                            </tr>
                                            
                                            <!-- Empty state for Vacant tab -->
                                            <tr class="empty-state vacant-state hidden">
                                                <td colspan="6" class="px-6 py-16">
                                                    <div class="text-center bg-green-50 rounded-lg p-8 max-w-2xl mx-auto">
                                                        <div class="inline-flex items-center justify-center h-16 w-16 rounded-full bg-green-100 mb-4">
                                                            <svg class="h-8 w-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
                                                            </svg>
                                                        </div>
                                                        <h3 class="text-lg font-medium text-gray-900">No Vacant Units</h3>
                                                        <p class="mt-2 text-sm text-gray-500">
                                                            All units in this property are currently occupied. When units become available, they will appear here.
                                                        </p>
                                                        <div class="mt-6">
                                                            <a href="{{ route('properties.units.create', $property->PropID) }}" class="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500">
                                                                <svg class="-ml-1 mr-2 h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                                    <path fill-rule="evenodd" d="M10 3a1 1 0 011 1v5h5a1 1 0 110 2h-5v5a1 1 0 11-2 0v-5H4a1 1 0 110-2h5V4a1 1 0 011-1z" clip-rule="evenodd" />
                                                                </svg>
                                                                Add More Units
                                                            </a>
                                                        </div>
                                                    </div>
                                                </td>
                                            </tr>
                                        @endforelse
                                    </tbody>
                                </table>
                                
                                <!-- Pagination -->
                                <div class="bg-white px-4 py-3 flex items-center justify-between border-t border-gray-200 sm:px-6">
                                    <div class="flex-1 flex justify-between sm:hidden">
                                        @if ($units->onFirstPage())
                                            <span class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white opacity-50 cursor-not-allowed">
                                                Previous
                                            </span>
                                        @else
                                            <a href="{{ $units->previousPageUrl() }}" class="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                                                Previous
                                            </a>
                                        @endif
                                        
                                        @if ($units->hasMorePages())
                                            <a href="{{ $units->nextPageUrl() }}" class="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                                                Next
                                            </a>
                                        @else
                                            <span class="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white opacity-50 cursor-not-allowed">
                                                Next
                                            </span>
                                        @endif
                                    </div>
                                    <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                                        <div>
                                            <p class="text-sm text-gray-700">
                                                Showing
                                                <span class="font-medium">{{ $units->firstItem() }}</span>
                                                to
                                                <span class="font-medium">{{ $units->lastItem() }}</span>
                                                of
                                                <span class="font-medium">{{ $units->total() }}</span>
                                                results
                                            </p>
                                        </div>
                                        <div>
                                            <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                                                {{ $units->links('pagination::tailwind') }}
                                            </nav>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection
